GenericScanner subclass: #ISCAS85Scanner
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'ISCAS85-Parser-Category'!


!ISCAS85Scanner methodsFor: 'scanning'!

overReadSpaces
	"Cariage returns, line feeds, spaces can be ignored."

	(nextChar = self endOfInput)
	ifFalse:[
		[ (nextChar = self endOfInput) not and:[(nextChar isSeparator) and:[ (nextChar = (Character cr)) not ] ] ] whileTrue:[ self getNextChar. ].].!

scanToken
	"Subclass must compute a token here.
	A Token must be an instance of GenericToken."

	| stream pos |
	self overReadSpaces.
	(nextChar = self endOfInput) ifTrue:[ ^EOIToken value: '' position: (self position)@(self position + 1) ].

	"Ignore comments"

	(nextChar = $*)
	ifTrue:[
		self getNextChar.
		[nextChar = (Character cr) or:[ nextChar = self endOfInput] ] whileFalse:[ self getNextChar ].
		(nextChar = (Character cr)) ifTrue:[ self getNextChar ].
		^self scanToken.].

	"Detect tokens"

	stream := ReadWriteStream on: ''.
	pos := self position.
	((nextChar isAlphabetic) or:[ #($< $> $- $: $$ $& $@) includes: nextChar ])
	ifTrue:[
		stream nextPut: nextChar.
		self getNextChar.
		[ nextChar isAlphaNumeric ] whileTrue:[ stream nextPut: nextChar. self getNextChar.].
		(self isKeyword: (stream contents))
		ifTrue:[^(KeywordToken new) value: (stream contents asUppercase); position: (pos@(self position)); yourself.]
		ifFalse:[^(IdToken new) value: (stream contents); position: (pos@(self position)); yourself.].]
	ifFalse:[
		(nextChar isDigit)
		ifTrue:[
			stream nextPut: nextChar.
			self getNextChar.
			[ nextChar isDigit] whileTrue:[ stream nextPut: nextChar. self getNextChar.].
			((nextChar isAlphabetic) or:[ #($< $> $- $: $$ $& $@) includes: nextChar ])
			ifTrue:[
				stream nextPut: nextChar.
				self getNextChar.
				[ nextChar isAlphaNumeric ] whileTrue:[ stream nextPut: nextChar. self getNextChar.].
				(self isKeyword: (stream contents))
				ifTrue:[^(KeywordToken new) value: (stream contents asUppercase); position: (pos@(self position)); yourself.]
				ifFalse:[^(IdToken new) value: (stream contents); position: (pos@(self position)); yourself.].]
			ifFalse:[
				^(IntToken new) value: (stream contents asNumber); position: (pos@(self position)); yourself.].]
		ifFalse:[
			(nextChar = (Character cr))
			ifTrue:[ 
				self getNextChar.
				^(EOLToken new) value: (stream contents); position: (pos@(self position)); yourself ].].].
	self getNextChar.
	^(ErrorToken new) value: (stream contents); position: (pos@(self position)); yourself.! !

!ISCAS85Scanner methodsFor: 'testing'!

isKeyword: aString
	"Returns true if aString contains a keyword."

	^#('INPT' 'AND' 'NAND' 'OR' 'NOR' 'XOR' 'XNOR' 'BUFF' 'NOT' 'FROM' '>SA0' '>SA1') includes: (aString asUppercase)! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ISCAS85Scanner class
	instanceVariableNames: ''!


!ISCAS85Scanner class methodsFor: 'initialize class'!

initialize
	"Initialize the class"
	"self initialize"

	self eoi: (EOIToken new value).
	self eoiToken: (EOIToken new).! !

GenericParser subclass: #ISCAS85Parser
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'ISCAS85-Parser-Category'!


!ISCAS85Parser methodsFor: 'private'!

scannerClass

	^ISCAS85Scanner! !

!ISCAS85Parser methodsFor: 'parsing'!

parse

	^self parseBenchmark: (ISCAS85Benchmark new)!

parseBenchmark: aBenchmark
	"Implements the parsing of an ISCAS'85 benchmark circuits.
	All nodes are stored in aBenchmark since node addresses must be unique."

	self parseBenchmark: aBenchmark actualNode: nil.
	^aBenchmark!

parseBenchmark: aBenchmark actualNode: aNode
	"Implements the parsing of an ISCAS'85 benchmark circuits.
	All nodes are stored in aBenchmark since node addresses must be unique."

	| actualNode |
	actualToken isEOIToken ifTrue:[ ^aNode ].
	actualNode := self parseBenchmarkLine: aBenchmark actualNode: aNode.
	^self parseBenchmark: aBenchmark actualNode: actualNode.!

parseBenchmarkLine: aBenchmark actualNode: aNode
	"Parses a Benchmark Line."

	| newNode input |
	newNode := ISCAS85Node new.
	(actualToken isIntToken)
	ifTrue:[
		newNode address: (actualToken value).
		self nextToken.
		(actualToken isIntToken)
		ifTrue:[ ^self parseFaninLine: aBenchmark actualNode: aNode context: (OrderedCollection with: (newNode address)) ]
		ifFalse:[
			(actualToken isIdToken)
			ifTrue:[
				newNode name: (actualToken value).
				self nextToken.
				(actualToken isKeywordToken)
				ifTrue:[
					newNode type: (actualToken value).
					self nextToken.
					(newNode type = 'FROM')
					ifFalse:[ ^self parseNodeLine: aBenchmark actualNode: newNode.]
					ifTrue:[ ^self parseFanoutLine: aBenchmark actualNode: aNode context: newNode ].]
				ifFalse:[ self error: 'A type must follow a name'.].]
			ifFalse:[
				(actualToken isEOLToken)
				ifTrue:[
					self nextToken.
					aNode isNil ifTrue:[ ^self error: 'Format Error!!!!!!!!'].
					input := aBenchmark at: (newNode address) ifAbsent:[ nil ].
					input isNil
					ifTrue:[ self error: 'Format fault !!!!!!!!' ]
					ifFalse:[
						aNode inputs add: input. 
						input outputs add: aNode. ].
					^aNode ]
				ifFalse:[ self error: 'A name or type must follow an address'.].].].]
	ifFalse:[ self error: 'A line must start with an integer'].!

parseFaninLine: aBenchmark actualNode: aNode context: aCollection

	| newNode |
	[(actualToken isIntToken)]
	whileTrue:[
		aCollection addLast: (actualToken value).
		self nextToken.].
	actualToken isEOLToken
	ifTrue:[
		self nextToken.
		aCollection do:[ :int | | input |
			input := aBenchmark at: int ifAbsent:[ nil ].
			input isNil
			ifTrue:[ self error: 'Format fault !!!!!!!!' ]
			ifFalse:[
				aNode inputs addLast: input. 
				input outputs addLast: aNode. ].].
		^aNode ]
	ifFalse:[
		(actualToken isKeywordToken and:[ aCollection size = 2 ])
		ifTrue:[
			newNode := ISCAS85Node new.
			newNode address: (aCollection at: 1).
			newNode name: ((aCollection at: 2) printString).
			newNode type: (actualToken value).
			self nextToken.
			(newNode type = 'FROM')
			ifFalse:[ ^self parseNodeLine: aBenchmark actualNode: newNode.]
			ifTrue:[ ^self parseFanoutLine: aBenchmark actualNode: aNode context: newNode ].]
		ifFalse:[ 
			self error: 'A Line must be finished by a cariage return character.' ].].!

parseFanoutLine: aBenchmark actualNode: aNode context: aFanoutNode

	| str |
	aNode isNil ifTrue:[ ^self error: 'Format error!!!!!!' ].

	((actualToken isIdToken) or:[ (actualToken isIntToken) ])
	ifTrue:[
		(actualToken isIntToken)
		ifTrue:[ str := actualToken value printString.]
		ifFalse:[ str := actualToken value].
		(str = (aNode name))
		ifTrue:[
			aFanoutNode inputs addLast: aNode.
			aNode outputs addLast: aFanoutNode.
			self nextToken.
			self parseFaults: aFanoutNode.
			aBenchmark at: (aFanoutNode address) put: aFanoutNode.
			^aNode. ]
		ifFalse:[ self error: 'Format error!!!!!!' ].]
	ifFalse:[ self error: 'A name must follow a FROM' ].!

parseFaults: aNode

	[(actualToken isKeywordToken)]
	whileTrue:[
		aNode faults add: (actualToken value).
		self nextToken ].
	actualToken isEOLToken
	ifTrue:[ self nextToken. ^aNode.]
	ifFalse:[ self error: 'A line must be finished with a cariage return character'].!

parseNodeLine: aBenchmark actualNode: aNode

	(actualToken isIntToken)
	ifTrue:[
		aNode fanout: (actualToken value).
		self nextToken.
		(actualToken isIntToken)
		ifTrue:[
			aNode fanin: (actualToken value).
			self nextToken.
			self parseFaults: aNode.
			aBenchmark at: (aNode address) put: aNode.
			^aNode. ]
		ifFalse:[ self error: 'A fanin must follow a fanout value' ].]
	ifFalse:[ self error: 'A fanout must follow a gate or port' ].! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ISCAS85Parser class
	instanceVariableNames: ''!


!ISCAS85Parser class methodsFor: 'initialization'!

initialize
	"self initialize"

	^super initialize! !

Object subclass: #ISCAS85Node
	instanceVariableNames: 'address name type fanout fanin faults inputs outputs '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'ISCAS85-Parser-Category'!


!ISCAS85Node methodsFor: 'accessing'!

address

	^address!

address: anInteger

	^address := anInteger!

fanin

	^fanin!

fanin: anInteger

	^fanin := anInteger!

fanout

	^fanout!

fanout: anInteger

	^fanout := anInteger!

faults

	^faults!

faults: aCollection

	^faults := aCollection!

inputs

	^inputs!

name

	^name!

name: aString

	^name := aString!

outputs

	^outputs!

type

	^type!

type: aString

	^type := aString! !

!ISCAS85Node methodsFor: 'initialize release'!

initialize

	address := 0.
	name := ''.
	type := ''.
	fanin := 0.
	fanout := 0.
	faults := OrderedCollection new.
	inputs := OrderedCollection new.
	outputs := OrderedCollection new.
	^self! !

!ISCAS85Node methodsFor: 'testing'!

isFrom

	^type = 'FROM'!

isInpt

	^type = 'INPT'! !

!ISCAS85Node methodsFor: 'printing'!

printOn: aStream

	address printOn: aStream.
	aStream nextPutAll: ' ', name.
	aStream nextPutAll: ' ', (type asString), ' '.
	fanout printOn: aStream.
	aStream nextPutAll: ' '.
	fanin printOn: aStream.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ISCAS85Node class
	instanceVariableNames: ''!


!ISCAS85Node class methodsFor: 'instance creation'!

new

	^super new initialize! !

Object subclass: #ISCAS85Benchmark
	instanceVariableNames: 'nodes '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'ISCAS85-Parser-Category'!


!ISCAS85Benchmark methodsFor: 'initialize release'!

initialize

	nodes := Dictionary new.
	^self! !

!ISCAS85Benchmark methodsFor: 'converting'!

asDTalkProgram
	"Converts self to a DTalk program."

	| stream |
	stream := ReadWriteStream on: ''.
	self asDTalkProgramOn: stream.
	^stream contents!

asDTalkProgramForPLSystems
	"Converts self to a DTalk program."

	| stream |
	stream := ReadWriteStream on: ''.
	self asDTalkProgramForPLSystemsOn: stream.
	^stream contents!

asDTalkProgramForPLSystemsOn: aStream
	"Converts self to a DTalk program."

	| cDict |
	cDict := Dictionary new.
	cDict
		at: 'INPT' put: PropositionalDiagnosisConnection;
		at: 'FROM' put: nil;
		at: 'AND' put: PLDCGenericAnd;
		at: 'NAND' put: PLDCGenericNand;
		at: 'OR' put: PLDCGenericOr;
		at: 'NOR' put: PLDCGenericNor;
		at: 'XOR' put: PLDCXor;
		at: 'XNOR' put: PLDCXnor;
		at: 'BUFF' put: PLDCIdentity;
		at: 'NOT' put: PLDCNot.
	self asDTalkProgramOn: aStream using: cDict and: 'PropositionalLogicDiagnosisSystem'.
	^self!

asDTalkProgramForTreeRelated
	"Converts self to a DTalk program."

	| stream |
	stream := ReadWriteStream on: ''.
	self asDTalkProgramForTreeRelatedOn: stream.
	^stream contents!

asDTalkProgramForTreeRelatedOn: aStream
	"Converts self to a DTalk program."

	| cDict |
	cDict := Dictionary new.
	cDict
		at: 'INPT' put: TreeRelatedConnection;
		at: 'FROM' put: nil;
		at: 'AND' put: TRCGenericAnd;
		at: 'NAND' put: TRCGenericNand;
		at: 'OR' put: TRCGenericOr;
		at: 'NOR' put: TRCGenericNor;
		at: 'XOR' put: TRCXor;
		at: 'XNOR' put: TRCXnor;
		at: 'BUFF' put: TRCIdentity;
		at: 'NOT' put: TRCNot.
	self asDTalkProgramOn: aStream using: cDict and: 'TreeRelatedDiagnosisSystem'.
	^self!

asDTalkProgramOn: aStream
	"Converts self to a DTalk program."

	| cDict |
	cDict := Dictionary new.
	cDict
		at: 'INPT' put: SConstraintVariable;
		at: 'FROM' put: nil;
		at: 'AND' put: SCCSIPAnd;
		at: 'NAND' put: SCCSIPNand;
		at: 'OR' put: SCCSIPOr;
		at: 'NOR' put: SCCSIPNor;
		at: 'XOR' put: SCCXor;
		at: 'XNOR' put: SCCXnor;
		at: 'BUFF' put: SCCIdentity;
		at: 'NOT' put: SCCNot.
	self asDTalkProgramOn: aStream using: cDict and: 'CPDiagnosisSystem'.
	^self!

asDTalkProgramOn: aStream using: aComponentDict and: aDiagnosisSystem
	"Converts self to a DTalk program."

	| sDict fromNodes compNodes index lc inNodes outNodes |
	inNodes := self inputs.
	outNodes := self outputs.
	sDict := Dictionary new.
	fromNodes := OrderedCollection new.
	compNodes := OrderedCollection new.
	lc := 1.
aStream nextPutAll: '"Declaring Connections..."
'.
	nodes do:[ :n |
		n isFrom
		ifTrue:[ fromNodes add: n.]
		ifFalse:[
			sDict at: n put: ('''s', (n address printString),'''').
			(inNodes includes: n)
			ifTrue:[ aStream nextPutAll: '	', aDiagnosisSystem, ' inputConnection: ', (sDict at: n), '.'.]
			ifFalse:[
				(outNodes includes: n)
				ifTrue:[ aStream nextPutAll: '	', aDiagnosisSystem, ' outputConnection: ', (sDict at: n), '.'.]
				ifFalse:[ aStream nextPutAll: '	', aDiagnosisSystem, ' connection: ', (sDict at: n), '.'.].].
			lc := self insertJunk: lc on: aStream.
			n isInpt
			ifFalse:[ compNodes add: n. ].].].
	fromNodes do:[ :n | sDict at: n put: (sDict at: (n inputs first)).].
	aStream nextPutAll: '
"Declaring Components..."
'.
	compNodes do:[ :n |
		((n fanin) > 2)
		ifTrue:[
			aStream nextPutAll: ('	', aDiagnosisSystem, ' component: ''', ((n type), (n address printString)),''' type: ', ((aComponentDict at: (n type)) printString), 
' argument: ', (n fanin printString), '.'). ]
		ifFalse:[
			aStream nextPutAll: ('	', aDiagnosisSystem, ' component: ''', ((n type), (n address printString)),''' type: ', ((aComponentDict at: (n type)) printString), '.').].
		lc := self insertJunk: lc on: aStream. ].
	aStream nextPutAll: '
"Declaring Connections..."
'.
	compNodes do:[ :n |
		(n inputs size = 2)
		ifTrue:[
			aStream nextPutAll: ('	', aDiagnosisSystem, ' connect: ', (sDict at: (n inputs at: 1)), ' withPort: #in1 at: ''', ((n type) , (n address printString)), '''.').
			lc := self insertJunk: lc on: aStream.
			aStream nextPutAll: ('	', aDiagnosisSystem, ' connect: ', (sDict at: (n inputs at: 2)), ' withPort: #in2 at: ''', ((n type) , (n address printString)), '''.').
			lc := self insertJunk: lc on: aStream. ]
		ifFalse:[
		(n inputs size = 1)
		ifTrue:[
			aStream nextPutAll: ('	', aDiagnosisSystem, ' connect: ', (sDict at: (n inputs at: 1)), ' withPort: #in at: ''', ((n type) , (n address printString)), '''.').
			lc := self insertJunk: lc on: aStream.]
		ifFalse:[
			index := 1.
			n inputs do:[ :in |
				aStream nextPutAll:
					('	', aDiagnosisSystem, ' connect: ', (sDict at: (in)), ' withPort: #in', (index printString),' at: ''', ((n type) , (n address printString)), '''.').
					lc := self insertJunk: lc on: aStream.
				index := index + 1.].].].
		aStream nextPutAll: ('	', aDiagnosisSystem, ' connect: ', (sDict at: n), ' withPort: #out at: ''', ((n type) , (n address printString)), '''.').
		lc := self insertJunk: lc on: aStream.].
	aStream nextPutAll: '
"Setting the Name..."
'.
	aStream nextPutAll: '	', aDiagnosisSystem, ' systemName: ''ISCAS85-Benchmark''.'.
	self insertJunk: lc on: aStream.
	^self!

insertJunk: aNumber on: aStream

	^self class insertJunk: aNumber on: aStream! !

!ISCAS85Benchmark methodsFor: 'accessing'!

at: anIndex

	^nodes at: anIndex!

at: anIndex ifAbsent: aBlock

	^nodes at: anIndex ifAbsent: aBlock!

at: anIndex put: aValue

	^nodes at: anIndex put: aValue!

inputs
	"Answers a set of nodes working as circuit input."

	^nodes select:[ :node | node isInpt ]!

inputsAndSignals
	"Answers a set of nodes working as circuit signals."

	| res |
	res := Set new: (nodes size).
	nodes do:[ :node |
		((node type = 'INPT') or:[ (node type = 'FROM') ])
		ifTrue:[ res add: node ].].
	^res!

nodes

	^nodes!

outputs
	"Answers a set of nodes working as circuit output."

	| res |
	res := Set new: (nodes size).
	nodes do:[ :node |
		(node outputs size = 0)
		ifTrue:[ res add: node ].].
	^res! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ISCAS85Benchmark class
	instanceVariableNames: ''!


!ISCAS85Benchmark class methodsFor: 'instance creation'!

new

	^super new initialize! !

!ISCAS85Benchmark class methodsFor: 'defaults'!

defaultBenchmarkDirectory

	^'/home/wotawa/MBD_BENCHMARKS/ISCAS85/DATA/'!

defaultDTalkDirectory

	^'/home/wotawa/SIEMENSMBD/programs/dtalkexamples/iscas85/'! !

!ISCAS85Benchmark class methodsFor: 'accessing'!

allBenchmarks

	^#( 'c17' 'c432' 'c499' 'c880' 'c1355' 'c1908' 'c2670' 'c3540' 'c5315' 'c6288' 'c7552')!

computeAllTestcasesFor: aBenchmark
	"Computes observations for the given benchmark circuit. It is assumed that this benchmark
	is still converted to a DTalk program. The result of this method is stored in several files."

	| bmDir dTalkDir file bm ds inputs outputs is index |
	bmDir := self defaultBenchmarkDirectory.
	dTalkDir := self defaultDTalkDirectory.
	file := bmDir , aBenchmark, '.isc'.
	bm := self loadBenchmark: file.
	ds := CPDiagnosisSystem loadSD: (dTalkDir , aBenchmark, '.st').
	is := bm inputs.
	inputs := Array new: (is size).
	index := 1.
	is do:[ :id |
		inputs at: index put: (ds connectionNamed: ('s', (id address printString))).
		index := index + 1. ].
	outputs := OrderedCollection new.
	bm outputs do:[ :id | outputs add: (ds connectionNamed: ('s' , (id address printString))) ].
	self computeAllTestcasesFor: aBenchmark usingTestcaseNr: (Array with: 1) andSD: ds andIndex: 1 andInputs: inputs andOBS: (Dictionary new) forOutputs: outputs.!

computeAllTestcasesFor: aBenchmark usingTestcaseNr: aNumber andSD: aDiagnosisSystem andIndex: anIndex andInputs: anArray andOBS: aDictionary forOutputs: aCollection

	| outOBS file stream num lc |
	((aNumber at: 1) > 100) ifTrue:[ ^self ].
	lc := 1.
	(anIndex > (anArray size))
	ifTrue:[
		num := aNumber at: 1.
		outOBS := aDiagnosisSystem computeOBSusingInputs: aDictionary andOutputs: aCollection.
		file := self defaultDTalkDirectory , aBenchmark, '/obs', aBenchmark, 'tc', (num printString), '.st'.
		stream := file asFilename writeStream.
		stream nextPutAll: '"Setting Inputs..."
'.
		aDictionary keysAndValuesDo:[ :conn :val |
			stream nextPutAll: '	CPDiagnosisSystem observe: ', (val printString) , ' on: ', ( conn named printString) , '.'.
			lc := self insertJunk: lc on: stream. ].
		stream nextPutAll: '
"Setting outputs..."
'.
		outOBS keysAndValuesDo:[ :conn :val |
			stream nextPutAll: '	CPDiagnosisSystem observe: ', (val printString) , ' on: ', ( conn named printString) , '.'.
			lc := self insertJunk: lc on: stream. ].
		stream close.
		aNumber at: 1 put: (num + 1).]
	ifFalse:[
		aDictionary at: (anArray at: anIndex) put: true.
		self
			computeAllTestcasesFor: aBenchmark
			usingTestcaseNr: aNumber
			andSD: aDiagnosisSystem
			andIndex: (anIndex + 1)
			andInputs: anArray
			andOBS: aDictionary
			forOutputs: aCollection.
		aDictionary at: (anArray at: anIndex) put: false.
		self
			computeAllTestcasesFor: aBenchmark
			usingTestcaseNr: aNumber
			andSD: aDiagnosisSystem
			andIndex: (anIndex + 1)
			andInputs: anArray
			andOBS: aDictionary
			forOutputs: aCollection.].!

computingBenchmarkResultsFor: aBenchmark on: aStream
	"Uses all testcases previously generated for aBenchmark to create benchmark results."

	| bmtcDir bm sdLoadTime ds bmtc obsLoadTime diagTime diags numDiags avgObsLoadTime avgDiagTime avgNumDiags num |
	bmtcDir := self defaultDTalkDirectory , aBenchmark, '/'.
	bm := self defaultDTalkDirectory , aBenchmark, '.st'.
	sdLoadTime := Time millisecondsToRun:[ ds := CPDiagnosisSystem loadSD: bm ].
	aStream
		nextPutAll: 'Performance result for
 
';
		nextPutAll: 'ISCAS85 Benchmark: ', bm, '
';
		nextPutAll: 'SD Load Time: ', (sdLoadTime printString), 'ms

';
		nextPutAll: '\begin{center}
\begin{tabular}{rrrr}
\hline
Testcase & OBS Loading Time & Diagnosis Time & Diagnoses \\
\hline
'.
	avgObsLoadTime := 0.0d.
	avgDiagTime := 0.0d.
	avgNumDiags := 0.0d.
	num := 0.
	bmtcDir asFilename directoryContents do: [ :file |
		num := num + 1.
		bmtc := bmtcDir, file.
		obsLoadTime := Time millisecondsToRun:[ ds loadOBS: bmtc ].
		diagTime := Time millisecondsToRun:[ diags := ds makeDiagnosis ].
		numDiags := diags size. 
		aStream
			nextPutAll: file, ' & ', (obsLoadTime printString) , ' & ', (diagTime printString) , ' & ', (numDiags printString) , ' \\
'.
		avgObsLoadTime := avgObsLoadTime + obsLoadTime.
		avgDiagTime := avgDiagTime + diagTime.
		avgNumDiags := avgNumDiags + numDiags.
		aStream flush.].
	(num = 0)
	ifFalse:[
		avgObsLoadTime := (avgObsLoadTime / num) ceiling asInteger.
		avgDiagTime := (avgDiagTime / num) ceiling asInteger.
		avgNumDiags := (avgNumDiags / num) ceiling asInteger.
		aStream
			nextPutAll: '\hline
';
			nextPutAll: 'Average & ', (avgObsLoadTime printString) , ' & ' ,(avgDiagTime printString) , ' & ' , (avgNumDiags printString) , ' \\
'.
		].
	aStream
		nextPutAll: '\hline
\end{tabular}
\end{center}

'.!

computingBenchmarkResultsWithFocusingFor: aBenchmark on: aStream
	"Uses all testcases previously generated for aBenchmark to create benchmark results."

	| bmtcDir bm sdLoadTime ds bmtc obsLoadTime diagTime diags numDiags avgObsLoadTime avgDiagTime avgNumDiags num |
	bmtcDir := self defaultDTalkDirectory , aBenchmark, '/'.
	bm := self defaultDTalkDirectory , aBenchmark, '.st'.
	sdLoadTime := Time millisecondsToRun:[ ds := CPDiagnosisSystem loadSD: bm ].
	aStream
		nextPutAll: 'Performance result for
 
';
		nextPutAll: 'ISCAS85 Benchmark: ', bm, '
';
		nextPutAll: 'SD Load Time: ', (sdLoadTime printString), 'ms

';
		nextPutAll: '\begin{center}
\begin{tabular}{rrrr}
\hline
Testcase & OBS Loading Time & Diagnosis Time & Diagnoses \\
\hline
'.
	avgObsLoadTime := 0.0d.
	avgDiagTime := 0.0d.
	avgNumDiags := 0.0d.
	num := 0.
	bmtcDir asFilename directoryContents do: [ :file |
		num := num + 1.
		bmtc := bmtcDir, file.
		obsLoadTime := Time millisecondsToRun:[ ds loadOBS: bmtc ].
Transcript show: 'Compute Focus ...'.
		diagTime := Time millisecondsToRun:[ ds focusOnDiscrepancies ].
Transcript show: 'Compute Diagnosis...
'.
		diagTime := diagTime + (Time millisecondsToRun:[ diags := ds makeDiagnosis ]).
		numDiags := diags size. 
		aStream
			nextPutAll: file, ' & ', (obsLoadTime printString) , ' & ', (diagTime printString) , ' & ', (numDiags printString) , ' \\
'.
		avgObsLoadTime := avgObsLoadTime + obsLoadTime.
		avgDiagTime := avgDiagTime + diagTime.
		avgNumDiags := avgNumDiags + numDiags.
		aStream flush.].
	(num = 0)
	ifFalse:[
		avgObsLoadTime := (avgObsLoadTime / num) ceiling asInteger.
		avgDiagTime := (avgDiagTime / num) ceiling asInteger.
		avgNumDiags := (avgNumDiags / num) ceiling asInteger.
		aStream
			nextPutAll: '\hline
';
			nextPutAll: 'Average & ', (avgObsLoadTime printString) , ' & ' ,(avgDiagTime printString) , ' & ' , (avgNumDiags printString) , ' \\
'.
		].
	aStream
		nextPutAll: '\hline
\end{tabular}
\end{center}

'.!

convertBenchmarksToDTalkPrograms
	"Converts all Benchmarks to DTalk programs."

	| bmDir dTalkDir file dFile |
	bmDir := self defaultBenchmarkDirectory.
	dTalkDir := self defaultDTalkDirectory.
	self allBenchmarks do:[ :fn |
		Transcript show: (fn , '.isc ').
		file := (bmDir , fn , '.isc' ) asFilename.
		dFile := (dTalkDir , fn , '.st') asFilename.
		self loadBenchmark: file convertToDTalkProgram: dFile ].
	^self!

insertJunk: aNumber on: aStream

	(aNumber > 100)
	ifTrue:[
		aStream nextPutAll: '!!
'.
		^1 ]
	ifFalse:[
		aStream nextPutAll: '
'.
		^aNumber + 1 ].!

loadBenchmark: aFilename
	"Creates a new benchmark using the contents of aFilename."

	| parser |
	parser := ISCAS85Parser new.
	^parser parse: (aFilename asFilename contentsOfEntireFile).!

loadBenchmark: aFilename convertToDTalkProgram: aDTalkFilename
	"Creates a new benchmark using the contents of aFilename."

	| parser stream bm |
	parser := ISCAS85Parser new.
	stream := aDTalkFilename asFilename writeStream.
	bm := parser parse: (aFilename asFilename contentsOfEntireFile).
	bm asDTalkProgramOn: stream.
	stream close.! !

GenericToken subclass: #EOLToken
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'ISCAS85-Parser-Category'!


!EOLToken methodsFor: 'testing'!

isEOLToken

	^true! !
ISCAS85Scanner initialize!

ISCAS85Parser initialize!


