Object subclass: #GenericToken
	instanceVariableNames: 'value position '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Generic-Parsing'!
GenericToken comment:
'This class implements the token object. Tokens are used by the
DDVScanner and DDVParser and store information about the
type, the value and the position.

type
	a token type stored as string

value
	the value of the token stored as string

position
	the position of the token within the source file
'!


!GenericToken methodsFor: 'accessing'!

position

	^position!

position: anInteger

	^position := anInteger!

value

	^value!

value: aValue

	^value := aValue! !

!GenericToken methodsFor: 'public'!

value: aValue position: aPosition
	"Sets the instance variables of self"

	self value: aValue.
	self position: aPosition.
	^self! !

!GenericToken methodsFor: 'testing'!

= aToken
	"Two tokens are equal if and only if they have the same type"

	^(aToken class) = (self class)!

isEmptyToken

	^false!

isEOIToken

	^false!

isEOLToken

	^false!

isErrorToken

	^false!

isIdToken

	^false!

isIntToken

	^false!

isKeywordToken

	^false! !

!GenericToken methodsFor: 'initialize-release'!

initialize

	^self! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GenericToken class
	instanceVariableNames: ''!


!GenericToken class methodsFor: 'instance-creation'!

new

	^super new initialize!

value: aValue
	"Creates an instance of a token and initializes the
	instance variables with the right values"

	^(super new) value: aValue position: 0!

value: aValue position: aPosition
	"Creates an instance of a token and initializes the
	instance variables with the right values"

	^(super new) value: aValue position: aPosition! !

GenericToken subclass: #KeywordToken
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Generic-Parsing'!


!KeywordToken methodsFor: 'initialize-release'!

initialize

	value := ''.
	^self! !

!KeywordToken methodsFor: 'testing'!

isKeywordToken

	^true! !

Array variableSubclass: #Stack
	instanceVariableNames: 'topPtr '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Generic-Parsing'!
Stack comment:
'=================================================
    Copyright (c) 1992 by Justin O. Graver.
    All rights reserved (with exceptions).
    For complete information evaluate "Object tgenCopyright."
=================================================

This class provides a more traditional push/pop interface for Arrays.'!


!Stack methodsFor: 'private'!

copyEmpty: aSize
	"Answer a copy of the receiver that contains no elements.

	This method should be redefined in subclasses that add
	instance variables, so that the state of those variables
	is preserved"

	^(super copyEmpty: aSize) topPtr: self topPtr.! !

!Stack methodsFor: 'state accessing'!

topPtr
	^topPtr!

topPtr: arg
	topPtr := arg! !

!Stack methodsFor: 'testing'!

isEmpty
	^topPtr = 0!

isFull
	^ topPtr = self basicSize! !

!Stack methodsFor: 'accessing'!

pop
	"Answer the object on top of the stack."

	| n |
	n := self at: topPtr.
	topPtr := topPtr - 1.
	^n!

pop: numElem 
	"Pop and discard top numElems and answer receiver"

	topPtr := topPtr - numElem!

push: anObject 
	"Push anObject onto the top of the stack."

	self isFull ifTrue: [self grow].
	topPtr := topPtr + 1.
	^self at: topPtr put: anObject!

size
	"Answer the number of objects on the stack."

	^topPtr!

top
	"Answer (without removing) the object on top of the stack."

	^self at: topPtr! !

!Stack methodsFor: 'initialize-release'!

initialize

	self topPtr: 0! !

!Stack methodsFor: 'enumerating'!

do: aBlock
	"Evaluate aBlock for each object on the stack, from top to bottom."

	^super reverseDo: aBlock!

reverseDo: aBlock
	"Evaluate aBlock for each object on the stack, from bottom to top."

	^super do: aBlock! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Stack class
	instanceVariableNames: ''!


!Stack class methodsFor: 'instance creation'!

new

	^self new: 100!

new: arg

	^( super new: arg ) initialize! !

GenericToken subclass: #EmptyToken
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Generic-Parsing'!


!EmptyToken methodsFor: 'initialize-release'!

initialize

	value := nil.
	^self! !

!EmptyToken methodsFor: 'testing'!

isEmptyToken

	^true! !

GenericToken subclass: #IdToken
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Generic-Parsing'!


!IdToken methodsFor: 'initialize-release'!

initialize

	value := ''.
	^self! !

!IdToken methodsFor: 'testing'!

isIdToken

	^true! !

GenericToken subclass: #ErrorToken
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Generic-Parsing'!


!ErrorToken methodsFor: 'initialize-release'!

initialize

	value := ''.
	^self! !

!ErrorToken methodsFor: 'testing'!

isErrorToken

	^true! !

Object subclass: #GenericScanner
	instanceVariableNames: 'source nextChar charBuffer '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Generic-Parsing'!
GenericScanner comment:
'(c) 1994 F. Wotawa, Technische Universitaet Wien, Institut fuer Informationssysteme

This generic class implements basic functions for scanners. The special scanning function
needed for a special programming language should be implemented by my subclasses.

Overwrite the following methods:

scanToken	Should answer the token used by the parser.'!


!GenericScanner methodsFor: 'scanning'!

delimiters
	"Answers a set of delimiters"

	^#( $# $, $) $( $[ $] ${ $} ) asSet!

endOfInput

	"Returns a char which represents the end of the input."

	^(self class eoi)!

endOfInputToken
	"Returns a token which represents the end of the input."

	^(self class eoiToken)!

getAllToken
	"Returns an array  containing all tokens."

	| acttoken list |
	self resetSource.
	list := OrderedCollection new.
	acttoken := self scanToken.
	[acttoken = (self endOfInputToken)]
	whileFalse:[
		list addLast: acttoken.
		acttoken := self scanToken.].
	list addLast: acttoken.
	^list asArray!

getNextChar
	"Gets a new character from source and stores this information
	in nextChar and charBuffer"

	self charBuffer nextPut: (self nextChar).
	self nextChar: (self source next).
	(self nextChar isNil)
	ifTrue:[
		self nextChar: (self endOfInput).]!

overReadSpaces
	"Cariage returns, line feeds, spaces can be ignored."

	(nextChar = self endOfInput)
	ifFalse:[
		[ (nextChar = self endOfInput) not and:[nextChar isSeparator ] ] whileTrue:[ self getNextChar. ].].!

position
	"Returns the actual position of the source"

	^(self source position)!

putBackChar
	"Do one step back on the stream source"

	(self charBuffer position > 1)
	ifTrue:[ self charBuffer position: (self charBuffer position - 1)].
	(self source position > 1)
	ifTrue:[self source position: (((self source) position) - 2)]
	ifFalse:[self source position: 0.].
	self nextChar: (self source next).!

scanSource: aString
	"Initialize the source and scan the string aString.
	Answere the first token scanned."

	self initSource: aString.
	^self scanToken!

scanToken
	"Subclass must compute a token here.
	A Token must be an instance of GenericToken."

	self subclassResponsibility! !

!GenericScanner methodsFor: 'private'!

emptyToken

	"Return the empty token. This token is stored befor the first
	call of scanToken has occured."

	^EmptyToken new! !

!GenericScanner methodsFor: 'initialize-release'!

initialize

	self charBuffer: (WriteStream on: (String new: 32)).
	^self!

initSource: aString

	"Sets the scanner source and other instance variables"

	self source: (ReadStream on: aString).
	self nextChar: self source next.
	self charBuffer reset.
	^self!

resetSource
	"Resets the source to allow a second scan"

	self source reset.
	self nextChar: (self source next).
	self charBuffer reset.
	^self! !

!GenericScanner methodsFor: 'testing'!

isKeyword: aString

	"Returns true if aString contains a key word.
	This methode must be overwritten by a subclass if used
	by scanToken."

	^false! !

!GenericScanner methodsFor: 'accessing'!

charBuffer

	^charBuffer!

charBuffer: aStream

	^charBuffer := aStream!

nextChar

	^nextChar!

nextChar: aChar

	^nextChar := aChar!

source

	^source!

source: aStream

	^source := aStream! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GenericScanner class
	instanceVariableNames: 'eoiToken eoi '!


!GenericScanner class methodsFor: 'accessing'!

eoi

	^eoi!

eoi: aChar

	^eoi := aChar!

eoiToken

	^eoiToken!

eoiToken: aDDVToken

	^eoiToken := aDDVToken! !

!GenericScanner class methodsFor: 'initialize class'!

initialize
	"Initialize the class"
	"GenericScanner initialize"

	self eoi: (EOIToken new value).
	self eoiToken: (EOIToken new).! !

!GenericScanner class methodsFor: 'instance creation'!

new

	^super new initialize!

scanFrom: aString

	^self new scanSource: aString! !

Object subclass: #GenericParser
	instanceVariableNames: 'scanner tokens actualToken failBlock requestor requestorRelativePosition errorPosition sourceFile '
	classVariableNames: 'DDVErrorSignal '
	poolDictionaries: ''
	category: 'Generic-Parsing'!
GenericParser comment:
'(c) 1994 by F.Wotawa, Technische Universitaet Wien, Institut fuer Informationssysteme

This class implements a generic parser. Special parser for particular programming languages
can use it''s basic methods.

The following methods should be overwritten by subclasses:

scannerClass		Should answer the scanner used for the actual programming
				language.

parse			Should contain the real parsing method.'!


!GenericParser methodsFor: 'public'!

parse: aString
	"Parse the string and returns the derivation stack or
	nil if an error occures."

	^self parse: aString ifFail:[nil]!

parse: aString ifFail: aBlock 

	self failBlock: aBlock.
	self exceptionHandlers
	handleDo:[
		self actualToken: (self scanner scanSource: aString).
		self tokens: (Stack new).
		^self parse.].
	^nil! !

!GenericParser methodsFor: 'parsing'!

parse
	"This method is the starting method of the parser"

	self subclassResponsibility! !

!GenericParser methodsFor: 'accessing'!

actualToken

	^actualToken!

actualToken: aToken

	^actualToken := aToken!

errorPosition

	^errorPosition!

errorPosition: anInteger

	^errorPosition := anInteger!

failBlock

	^failBlock!

failBlock: aBlock

	^failBlock := aBlock!

requestor

	^requestor!

requestor: aWrapper

	^requestor := aWrapper!

requestorRelativePosition: aValue

	^requestorRelativePosition := aValue!

scanner

	^scanner!

scanner: aDDVScanner

	^scanner := aDDVScanner!

sourceFile

	^sourceFile!

sourceFile: aString

	^sourceFile := aString!

tokens

	^tokens!

tokens: aTokenArray

	^tokens := aTokenArray! !

!GenericParser methodsFor: 'scanning'!

getActualToken
	"Answers the actualToken"

	^self actualToken!

nextToken
	"Calculates a new token."

	(self tokens isEmpty)
	ifTrue:[
		self actualToken: self scanner scanToken.]
	ifFalse:[
		self actualToken: self tokens pop.].!

prevToken: aToken
	"Stores aToken into tokens for later usage"

	self tokens push: aToken.
	self actualToken: aToken.! !

!GenericParser methodsFor: 'initialize-release'!

initialize

	tokens := ( Stack new: 50 ).
	scanner := self scannerClass new.
	requestorRelativePosition := 0.
	actualToken := nil.
	^self! !

!GenericParser methodsFor: 'private'!

error: aString
	"Raises the error signal."

	actualToken isNil
	ifTrue:[ self error: aString atPosition: 0@0 ]
	ifFalse:[ self error: aString atPosition: (actualToken position) ].!

error: aString atPosition: aPos
	"Raises the error signal."

	self errorPosition: (requestorRelativePosition + aPos).
	self raiseDDVErrorSignal: aString.!

scannerClass

	"This methode should be overriden by subclasses"

	^GenericScanner! !

!GenericParser methodsFor: 'exception handling'!

abort
	| block |

	block := self failBlock.
	self failBlock: nil.
	^block value!

exceptionHandlers
	"Answer a HandlerCollection that will catch and handle scanner and parser errors."

	| handlers |
	handlers := HandlerList new.
	handlers on: self class ddvErrorSignal
		handle: [:ex | 
			(self requestor notNil)
			ifTrue:[ | txt |
				txt := '<- ERROR: ', ex localErrorString.
				self requestor widget controller insertAndSelect: txt at: (self errorPosition y).
				self requestor widget controller selectAndScrollFrom: (self errorPosition y) to: (self errorPosition y + (txt size) - 1).
				self requestor takeKeyboardFocus.]
			ifFalse:[
				Dialog warn: ex localErrorString ,' at position ', (self errorPosition x printString) , '
'.].
			self abort].

	^handlers!

raiseDDVErrorSignal: aString 

	self class ddvErrorSignal raiseWith: nil errorString: aString! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GenericParser class
	instanceVariableNames: ''!


!GenericParser class methodsFor: 'initialization'!

initialize
	"GenericParser initialize"

	self ddvErrorSignal: ( Signal new nameClass: self message: #parseErrorOccured )! !

!GenericParser class methodsFor: 'instance creation'!

new

	^super new initialize! !

!GenericParser class methodsFor: 'accessing'!

ddvErrorSignal
	"GenericParser ddvErrorSignal"

	^DDVErrorSignal!

ddvErrorSignal: aSignal

	^DDVErrorSignal := aSignal! !

GenericToken subclass: #EOIToken
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Generic-Parsing'!


!EOIToken methodsFor: 'initialize-release'!

initialize

	position := 0.
	value := Character value: 4.
	^self! !

!EOIToken methodsFor: 'testing'!

isEOIToken

	^true! !

GenericToken subclass: #IntToken
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Generic-Parsing'!


!IntToken methodsFor: 'initialize-release'!

initialize

	value := 0.
	^self! !

!IntToken methodsFor: 'testing'!

isIntToken

	^true! !
GenericScanner initialize!

GenericParser initialize!


