DiagnosisObject subclass: #TreeRelatedConnection
	instanceVariableNames: 'from to value observedValue checked diagnosisSystem diagStore '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TreeRelatedConnection comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements a diagnosis connection. Every connection is connected with one output
port  and any number of inputs. Note that the TREE algorithm only works correctly if the underlying
system is tree structured!!

Instance variables:

from		The component where self is connected with the #out port
to		A set of input ports where self is connected with.
value	The computed value
observedValue		An observation
checked 			This variable is used by TREE to indicate that self has been
					visited previously during computation.
diagnosisSystem 	A link to my diagnosis system.'!


!TreeRelatedConnection methodsFor: 'adding-removing'!

addPort: aPort
	"Adds a component port to my to set."

	^to add: aPort! !

!TreeRelatedConnection methodsFor: 'diagnosis'!

diagTopDown: aValue
	"Implements the search part of the TREE algorithm"

	checked
	ifTrue:[
		(value = aValue)
		ifTrue:[ ^DiagnoseSet with: (Diagnose new) ]
		ifFalse:[ ^DiagnoseSet new ].]
	ifFalse:[
		checked := true.
		(value = aValue)
		ifTrue:[ ^DiagnoseSet with: (Diagnose new) ]
		ifFalse:[ ^from diagTopDown: aValue ].].!

evaluateTopDown
	"Implements the evaluation function."

	value isNil ifFalse:[ ^self ].
	self isInput
	ifTrue:[
		value := observedValue.
		checked := true.]
	ifFalse:[
		value := from evaluateTopDown. ].!

forwardCheck: aValue
	"Answers true if the value does not contradict my observations or
	the observations of connections depending on me."

	(self hasObservation)
	ifTrue:[ (aValue = observedValue) ifFalse:[ ^false ].].
	(aValue = value) ifTrue:[ ^true ].
	to do:[ :port |
			(port component forwardCheck: aValue for: (port identifier))
			ifFalse:[ ^false ].].
	^true!

forwardCheck: aValue except: aComponent
	"Answers true if the value does not contradict my observations or
	the observations of connections depending on me."

	(self hasObservation)
	ifTrue:[ (aValue = observedValue) ifFalse:[ ^false ].].
	to do:[ :port |
			(port component = aComponent)
			ifFalse:[
				(port component forwardCheck: aValue for: (port identifier))
				ifFalse:[ ^false ].].].
	^true!

modifiedDiagTopDown: aValue
	"Implements the search part of the TREE algorithm"

	checked
	ifTrue:[
		(value = aValue)
		ifTrue:[ ^DiagnoseSet with: (Diagnose new) ]
		ifFalse:[ ^DiagnoseSet new ].]
	ifFalse:[
		checked := true.
		(value = aValue)
		ifTrue:[ ^DiagnoseSet with: (Diagnose new) ]
		ifFalse:[ 
			(diagStore includesKey: aValue)
			ifFalse:[ diagStore at: aValue put: (from modifiedDiagTopDown: aValue).].
			^diagStore at: aValue ].].!

modifiedTreeDiagTopDown
	"Implements a part of the TREE algorithm"

	| diags |
	self isInput
	ifTrue:[
		value := observedValue.
		checked := true.]
	ifFalse:[
		value := from modifiedTreeDiagTopDown.
		(self hasObservation)
		ifTrue:[
			(value = observedValue)
			 ifFalse:[
				(diagStore includesKey: observedValue)
				ifTrue:[ diags := diagStore at: observedValue ]
				ifFalse:[
					diags := self modifiedDiagTopDown: observedValue.
					diagStore at: observedValue put: diags.].
				diagnosisSystem diagnoses: (
					((diagnosisSystem diagnoses) times: 
					diags) reduceDiagnosisSize: (diagnosisSystem maxDiagnosisSize)).].].].!

resetValues
	"Set the values to nil."

	value := nil.
	observedValue := nil.
	checked := false.
	diagStore := Dictionary new.
	^self!

treeDiagTopDown
	"Implements a part of the TREE algorithm"

	self isInput
	ifTrue:[
		value := observedValue.
		checked := true.]
	ifFalse:[
		value := from treeDiagTopDown.
		(self hasObservation)
		ifTrue:[
			(value = observedValue)
			 ifFalse:[ 
				diagnosisSystem diagnoses: (
					((diagnosisSystem diagnoses) times: 
					(self diagTopDown: observedValue)) reduceDiagnosisSize: (diagnosisSystem maxDiagnosisSize)).].].].! !

!TreeRelatedConnection methodsFor: 'accessing'!

diagnosisSystem

	^diagnosisSystem!

diagnosisSystem: aDiagSystem

	^diagnosisSystem := aDiagSystem!

from

	^from!

from: aComponent

	^from := aComponent!

observedValue

	^observedValue!

setObservation: aValue

	^observedValue := aValue!

setValue: aValue

	^value := aValue!

to

	^to!

value

	^value! !

!TreeRelatedConnection methodsFor: 'testing'!

hasObservation

	^observedValue isNil not!

hasValue

	^value isNil not!

isConsistent
	"Answers true if my value does not contradict the stored observation."

	(self hasValue and:[ self hasObservation ])
	ifTrue:[ ^value = observedValue ]
	ifFalse:[ ^true ].!

isInput

	^from isNil! !

!TreeRelatedConnection methodsFor: 'initialize-release'!

initialize

	super initialize.
	from := nil.
	to := OrderedCollection new.
	value := nil.
	observedValue := nil.
	checked := false.
	diagStore := Dictionary new.
	^self! !

!TreeRelatedConnection methodsFor: 'printing'!

printOn: aStream

	aStream nextPutAll: (identifier printString), ' '.! !

!TreeRelatedConnection methodsFor: 'converting'!

convertToTree: aDiagnosisSystem
	"Converts my connected system in a tree."

	| conn |
	conn := TreeRelatedConnection identifier: ('s', (aDiagnosisSystem nextNumber printString)).
	from isNil
	ifTrue:[ aDiagnosisSystem at: self addInputConnection: conn ]
	ifFalse:[
		aDiagnosisSystem at: self addConnection: conn.
		from convertToTree: aDiagnosisSystem with: conn ].
	^conn!

convertToTreeStartingPoint: aDiagnosisSystem
	"Converts my connected system in a tree."

	| conn |
	conn := TreeRelatedConnection identifier: ('s', (aDiagnosisSystem nextNumber printString)).
	aDiagnosisSystem at: self addOutputConnection: conn.
	from convertToTree: aDiagnosisSystem with: conn.
	^conn! !

GenericDiagnosisSystem subclass: #TreeRelatedDiagnosisSystem
	instanceVariableNames: 'inputs outputs '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TreeRelatedDiagnosisSystem comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements a diagnosis system containing out of components and connections
and the TREE algorithm.

Instance variables:

inputs			A set of input connections
outputs			A set of output connections
connections		A dictionary associating connection names to connection objects
components		A dictionary associating component names to component objects
observations		A dictionary associating connections to values
diagnoses		A DiagnoseSet storing the previously computed diagnoses
maxDiagnosisSize	A number setting the maximum diagnosis size.

A TreeRelatedDiagnosisSystem allows to load and save observations and system descriptions.
The structure of the stored diagnosis system is not fixed. To correctly use the TREE algorithm
only tree structured system should be used.

This implementation has been tested using and-or-trees. The following runtime results has
been obtained on an ULTRASPARC 150 MHz, with 128 MB main memory under Solaris.

CONST	MIN		MEAN	MEDIAN	MEDIAN0.8	MEDIAN0.9		MAX	VAR
1.0		0.0		0.5845	1.0			1.0			1.0				3.0		0.270995
2.0		1.0		1.727	2.0			2.0			2.0				8.0		0.324629
3.0		2.0		4.731	4.0			7.0			7.0				69.0	6.99104
4.0		5.0		9.8115	8.0			13.0		15.0			23.0	14.514
5.0		10.0	30.344	20.0		49.0		54.0			86.0	298.82
6.0		15.0	53.584	54.0		79.0		91.0			154.0	743.667
7.0		32.0	132.675	154.0		181.0		198.0			348.0	3539.81
8.0		52.0	270.879	276.0		361.0		391.0			675.0	10607.1
9.0		129.0	612.585	641.0		738.0		791.0			1193.0	28210.8
10.0	184.0	1170.37	1208.0		1365.0		1445.0			2824.0	64730.9
'!


!TreeRelatedDiagnosisSystem methodsFor: 'accessing'!

componentNamed: anIdentifier
	"Answers the component with the given name. If no such component exists
	nil is returned."

	^components at: anIdentifier ifAbsent:[ nil ]!

connectionNamed: anIdentifier
	"Answers the connection with the given name. If no such connection exists
	nil is returned."

	^connections at: anIdentifier ifAbsent:[ nil ]!

inputs

	^inputs!

observationValue: aConnection
	"Returns the observed value associated with the given connection.
	nil is returned if no such observation exists."

	^observations at: aConnection ifAbsent:[ nil ]!

outputs

	^outputs! !

!TreeRelatedDiagnosisSystem methodsFor: 'initialize-release'!

initialize

	super initialize.
	inputs := Set new.
	outputs := Set new.
	connections := Dictionary new.
	components := Dictionary new.
	diagnoses := DiagnoseSet new.
	observations := Dictionary new.
	maxDiagnosisSize := 1.
	^self! !

!TreeRelatedDiagnosisSystem methodsFor: 'adding-removing'!

addComponent: aComponent
	"Adds the component to self."

	aComponent diagnosisSystem: self.
	^components at: (aComponent identifier) put: aComponent!

addConnection: aConnection
	"Adds the connection to self."

	aConnection diagnosisSystem: self.
	^connections at: (aConnection identifier) put: aConnection!

addInput: aConnection
	"Adds the connection as input to self."

	inputs add: aConnection.
	^self addConnection: aConnection!

addOutput: aConnection
	"Adds the connection as output to self."

	outputs add: aConnection.
	^self addConnection: aConnection!

observation: aValue on: aConnection
	"Adds an observation to the given connection."

	^observations at: aConnection put: aValue!

removeOBS
	"Removes the observations"

	^observations := Dictionary new!

removeOBS: aConnection
	"Removes the observation related to aConnection"

	^observations removeKey: aConnection ifAbsent:[ nil ]! !

!TreeRelatedDiagnosisSystem methodsFor: 'private'!

diagnoses: aDiagnosisSet

	^diagnoses := aDiagnosisSet! !

!TreeRelatedDiagnosisSystem methodsFor: 'converting'!

convertToTree
	"For every output a tree structured system is created which is tree structured
	and implements the same functionality. A dictionary of tree structured system
	is answered as result."

	| result |
	result := Dictionary new.
	outputs do:[ :outConn |
		result at: (outConn identifier) put: (self convertToTree: outConn) ].
	^result!

convertToTree: anOutputConnection
	"Creates a new tree structured diagnosis system for the given output
	connection with the same functionality."

	| result |
	result := TRDiagnosisSystemWithAssociations new.
	result identifier: (identifier, '-', (anOutputConnection identifier asString)).
	anOutputConnection convertToTreeStartingPoint: result.
	result loadRelatedObservations: observations.
	^result! !

!TreeRelatedDiagnosisSystem methodsFor: 'diagnosis'!

evaluate
	"Evaluates the given system using the observations stored in self."

	self resetSystem.
	self setObservations.
	outputs do:[ :outConn | outConn evaluateTopDown ].
	^self!

makeDiagnosis

	| diags |
	diags := DiagnoseSet with: (Diagnose new).
	self resetSystem.
	self setObservations.
	self evaluate.
	outputs do:[ :conn |
		conn isConsistent ifFalse:[
			diagnoses := DiagnoseSet with: (Diagnose new).
			conn modifiedTreeDiagTopDown.
			diags := (diags times: diagnoses) minimize reduceDiagnosisSize: maxDiagnosisSize.].].
	^diagnoses := diags!

maxDiagnosisSize

	^maxDiagnosisSize!

maxDiagnosisSize: aValue

	^maxDiagnosisSize := aValue!

resetSystem
	"Sets all values to nil."

	connections do:[ :conn | conn resetValues ].
	components do:[ :comp | comp resetValues ].!

setObservations
	"Sets the connections to the given values."

	observations keysAndValuesDo:[ :conn :val |
		conn setObservation: val.].!

treeDiag
	"Implements the TREE diagnosis algorithm for tree structured systems."

	(outputs size = 1) ifFalse:[ ^self error: 'A tree structured system has exactly one output' ].
	self resetSystem.
	self setObservations.
	diagnoses := DiagnoseSet with: (Diagnose new).
	outputs do:[ :outConn | outConn treeDiagTopDown ].
	^diagnoses! !

!TreeRelatedDiagnosisSystem methodsFor: 'public-accessing'!

component: anIdentifier type: aCompType

	self class actualSystem: self.
	self class component: anIdentifier type: aCompType.!

component: anIdentifier type: aCompType argument: anArgument

	self class actualSystem: self.
	self class component: anIdentifier type: aCompType argument: anArgument.!

connect: aConnId withPort: aPortId at: aCompId

	self class actualSystem: self.
	self class connect: aConnId withPort: aPortId at: aCompId.!

connection: anIdentifier

	self class actualSystem: self.
	self class connection: anIdentifier.!

inputConnection: anIdentifier

	self class actualSystem: self.
	self class inputConnection: anIdentifier.!

observe: aValue on: aConnId

	self class actualSystem: self.
	self class observe: aValue on: aConnId.!

outputConnection: anIdentifier

	self class actualSystem: self.
	self class outputConnection: anIdentifier.!

systemName: anIdentifier

	self class actualSystem: self.
	self class systemName: anIdentifier.! !

!TreeRelatedDiagnosisSystem methodsFor: 'printing'!

printOBSOn: aStream

	| lines |
	lines := 0.
	observations keysAndValuesDo:[ :conn :val |
		aStream nextPutAll: (self class printString), ' observe: '.
		aStream nextPutAll: (val printString).
		aStream nextPutAll: ' on: '.
		aStream nextPutAll: (conn identifier printString), '.'.
		lines := self insertJunk: lines on: aStream.].!

printOn: aStream

	aStream nextPutAll: (identifier printString), ' '.!

printSDOn: aStream

	| lines |
	lines := 0.
	aStream nextPutAll: (self class printString), ' systemName: '.
	aStream nextPutAll: (identifier printString) , '.'.
	lines := self insertJunk: lines on: aStream.

	inputs do:[ :conn |
		aStream nextPutAll: (self class printString), ' inputConnection: '.
		aStream nextPutAll: (conn identifier printString), '.'.
		lines := self insertJunk: lines on: aStream.].
	outputs do:[ :conn |
		aStream nextPutAll: (self class printString), ' outputConnection: '.
		aStream nextPutAll: (conn identifier printString) , '.'.
		lines := self insertJunk: lines on: aStream.].
	connections do:[ :conn |
		((inputs includes: conn) or: [ outputs includes: conn ])
		ifFalse:[
			aStream nextPutAll: (self class printString), ' connection: '.
			aStream nextPutAll: (conn identifier printString), '.'.
			lines := self insertJunk: lines on: aStream.].].

	components do:[ :comp |
		aStream nextPutAll: (self class printString), ' component: ', (comp identifier printString) , ' type: ', (comp class printString), '.'.
		lines := self insertJunk: lines on: aStream.].

	components do:[ :comp |
		comp defaultPorts do:[ :pid |
			aStream nextPutAll: (self class printString), ' connect: '.
			aStream nextPutAll: ((comp inputs at: pid) printString).
			aStream nextPutAll: ' withPort: ', (pid printString), ' at: ', (comp identifier printString), '.'.
			lines := self insertJunk: lines on: aStream. ].
		aStream nextPutAll: (self class printString), ' connect: '.
		aStream nextPutAll: ((comp output) printString).
		aStream nextPutAll: ' withPort: #out at: ', (comp identifier printString), '.'.
		lines := self insertJunk: lines on: aStream.].! !

!TreeRelatedDiagnosisSystem methodsFor: 'generate-examples'!

booleanTestcaseGenerator
	"Generates test case examples. It is assumed that only boolean values are used."

	| rand outArray num |
	observations := Dictionary new.
	rand := Random new.
	rand next. rand next. rand next. rand next.
	inputs do:[ :conn | self observation: (rand next <= 0.5) on: conn ].
	self evaluate.
	outArray := outputs asArray.
	num := ((rand next * (outArray size - 1)) + 1) rounded.
	1 to: (outArray size) do:[ :index |
		(index = num)
		ifTrue:[ self observation: ((outArray at: index) value not) on: (outArray at: index).]
		ifFalse:[ self observation: ((outArray at: index) value) on: (outArray at: index).].].
	^self! !

!TreeRelatedDiagnosisSystem methodsFor: 'file-accessing'!

loadOBS: aFilename
	"Loads the observation stored in aFilename."

	TreeRelatedDiagnosisSystem actualSystem: self.
	TreeRelatedDiagnosisSystem loadOBS: aFilename.
	^self!

loadSD: aFilename
	"Load a new system description."

	^self class loadSD: aFilename!

saveOBS: aFilename
	"Save the observation stored in self into the file aFilename."

	| stream |
	stream := aFilename asFilename writeStream.
	self printOBSOn: stream.
	stream close.
	^self!

saveSD: aFilename
	"Save the system stored in self into the file aFilename."

	| stream |
	stream := aFilename asFilename writeStream.
	self printSDOn: stream.
	stream close.
	^self! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

TreeRelatedDiagnosisSystem class
	instanceVariableNames: ''!


!TreeRelatedDiagnosisSystem class methodsFor: 'file-accessing'!

loadOBS: aFilename
	"Loads observations stored in aFilename into my actual instance."

	(aFilename asFilename exists)
	ifTrue:[
		ActualSystem isNil
		ifTrue:[ self error: 'An actual system must exist before assigning observations to it.' ]
		ifFalse:[
			ActualSystem removeOBS. 
			aFilename asFilename fileIn. ].]
	ifFalse:[ self error: 'The file ', (aFilename printString), ' does not exist or is not readable' ].!

loadSD: aFilename
	"Loads a new system stored in aFilename. A new instance of self is created."

	(aFilename asFilename exists)
	ifTrue:[
		TreeRelatedDiagnosisSystem actualSystem: (self new).
		aFilename asFilename fileIn. ]
	ifFalse:[ self error: 'The file ', (aFilename printString), ' does not exist or is not readable' ].
	^TreeRelatedDiagnosisSystem actualSystem! !

!TreeRelatedDiagnosisSystem class methodsFor: 'file-accessing-private'!

component: anIdentifier type: aComponentType

	| obj |
	ActualSystem isNil
	ifFalse:[
		obj := aComponentType identifier: anIdentifier.
		ActualSystem addComponent: obj.].!

component: anIdentifier type: aComponentType argument: anArgument

	| obj |
	ActualSystem isNil
	ifFalse:[
		obj := aComponentType identifier: anIdentifier with: anArgument.
		ActualSystem addComponent: obj.].!

connect: aConnId withPort: aPortId at: aCompId

	| conn comp |
	ActualSystem isNil
	ifFalse:[
		conn := ActualSystem connectionNamed: aConnId.
		comp := ActualSystem componentNamed: aCompId.
		conn isNil ifTrue:[ ^self error: 'Connection ', (aConnId printString), ' does not exist!! ' ].
		comp isNil ifTrue:[ ^self error: 'Component ', (aCompId printString), ' does not exist!! ' ].
		comp connect: conn usingPort: aPortId. ].!

connection: anIdentifier

	| obj |
	ActualSystem isNil
	ifFalse:[
		obj := TreeRelatedConnection identifier: anIdentifier.
		ActualSystem addConnection: obj.].!

inputConnection: anIdentifier

	| obj |
	ActualSystem isNil
	ifFalse:[
		obj := TreeRelatedConnection identifier: anIdentifier.
		ActualSystem addInput: obj.].!

maxDiagnosisSize: aValue

	ActualSystem isNil
	ifFalse:[ ActualSystem maxDiagnosisSize: aValue.].!

observe: aValue on: aConnId

	| obj |
	ActualSystem isNil
	ifFalse:[
		obj := ActualSystem connectionNamed: aConnId.
		ActualSystem observation: aValue on: obj.].!

outputConnection: anIdentifier

	| obj |
	ActualSystem isNil
	ifFalse:[
		obj := TreeRelatedConnection identifier: anIdentifier.
		ActualSystem addOutput: obj.].!

systemName: anIdentifier

	ActualSystem isNil
	ifFalse:[ ActualSystem identifier: anIdentifier.].! !

!TreeRelatedDiagnosisSystem class methodsFor: 'generate-examples'!

dechterEG: anInputArray storeIn: aDiagSystem with: andLevel with: aNumber

	| comp num inputs conn |
	(anInputArray size = 1)
	ifTrue:[ ^aDiagSystem ]
	ifFalse:[
		num := aNumber.
		inputs := Array new: ((anInputArray size) / 2) rounded.
		1 to: (inputs size) do:[ :index |
			andLevel
			ifTrue:[ comp := TRCAnd identifier: ('A', (num printString)). ]
			ifFalse:[ comp := TRCOr identifier: ('O', (num printString)) ].
			aDiagSystem addComponent: comp.
			comp connect: (anInputArray at: ((2*index) - 1)) usingPort: #in1.
			comp connect: (anInputArray at: (2*index)) usingPort: #in2.
			(inputs size = 1)
			ifTrue:[
				conn := TreeRelatedConnection identifier: 'O'.
				aDiagSystem addOutput: conn.]
			ifFalse:[
				conn := TreeRelatedConnection identifier: ('S', ((aNumber + index - 1) printString)).
				aDiagSystem addConnection: conn.].
			inputs at: index put: conn.
			comp connect: conn usingPort: #out.
			num := num + 1.].
		^self dechterEG: inputs storeIn: aDiagSystem with: (andLevel not) with: num.].!

dechterExampleGenerator: aConstant
	"Generates a tree structured diagnosis system."

	| rand |
	rand := Random new.
	rand next.
	rand next.
	^self dechterExampleGenerator: aConstant startingWithAnd: (rand next <= 0.5)!

dechterExampleGenerator: aConstant startingWithAnd: aBoolean
	"Generates a tree structured diagnosis system."

	| inputs diagSystem conn |
	diagSystem := TreeRelatedDiagnosisSystem identifier: 'DechterExample'.
	inputs := Array new: (2 ** aConstant).
	1 to: (inputs size) do:[ :index |
		conn := TreeRelatedConnection identifier: ('I', (index printString)).
		diagSystem addInput: conn.
		inputs at: index put: conn.].
	self dechterEG: inputs storeIn: diagSystem with: aBoolean with: 1.
	^diagSystem!

doDechterPerformanceTestOn: aStream
	"Generates dechter examples and checks performance."

	| maxC maxTrials ds t diags |
	maxC := 10. maxTrials := 1000.
	aStream nextPutAll: 'constant	time	diags'; cr.
	1 to: maxC do:[ :index |
		Transcript show: (index printString), ' of ', (maxC printString), '
'.
		ds := self dechterExampleGenerator: index startingWithAnd: true.
		1 to: maxTrials do:[ :tindex |
			ds booleanTestcaseGenerator.
			t := Time millisecondsToRun: [ diags := ds treeDiag ].
			aStream nextPutAll: (index printString), '	', (t printString), '	', (diags size printString).
			aStream cr.].
		aStream flush.
		ds := self dechterExampleGenerator: index startingWithAnd: false.
		1 to: maxTrials do:[ :tindex |
			ds booleanTestcaseGenerator.
			t := Time millisecondsToRun: [ diags := ds treeDiag ].
			aStream nextPutAll: (index printString), '	', (t printString), '	', (diags size printString).
			aStream cr.].
		aStream flush.].!

startRuntimeTest

	" TreeRelatedDiagnosisSystem startRuntimeTest"

	| file stream |
	file := 'runtime.tree.txt' asFilename.
	stream := file writeStream.
	self doDechterPerformanceTestOn: stream.
	stream close.!

startSystemPerformanceTest
	"This method implements a computer performance test to allow to compare
	runs maden using different computer systems."

	" TreeRelatedDiagnosisSystem startSystemPerformanceTest"

	| file stream maxC maxTrials ds t diags avgTime |

	file := 'performancetest.txt' asFilename.
	stream := file writeStream.
	maxC := 8. maxTrials := 50.
	avgTime := 0.
	stream nextPutAll: 'constant	time	diags'; cr.
	ds := self dechterExampleGenerator: maxC startingWithAnd: true.
	1 to: maxTrials do:[ :tindex |
		ds booleanTestcaseGenerator.
		t := Time millisecondsToRun: [ diags := ds treeDiag ].
		avgTime := avgTime + t.
		stream nextPutAll: (maxC printString), '	', (t printString), '	', (diags size printString).
		stream cr.].
	stream flush.
	ds := self dechterExampleGenerator: maxC startingWithAnd: false.
	1 to: maxTrials do:[ :tindex |
		ds booleanTestcaseGenerator.
		t := Time millisecondsToRun: [ diags := ds treeDiag ].
		avgTime := avgTime + t.
		stream nextPutAll: (maxC printString), '	', (t printString), '	', (diags size printString).
		stream cr.].
	stream flush.
	stream nextPutAll: ('AVG	', (((avgTime / (2*maxTrials)) ceiling) printString), '	n.A.').
	stream cr.
	stream close.! !

TreeRelatedDiagnosisSystem subclass: #TRDiagnosisSystemWithAssociations
	instanceVariableNames: 'connAssocs compAssocs number '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TRDiagnosisSystemWithAssociations comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements a diagnosis system like TreeRelatedDiagnosisSystem. Additionally
associations of names to connections and components can be specified. Instances of this
class are used as result for tree conversion of more general systems.'!


!TRDiagnosisSystemWithAssociations methodsFor: 'initialize-release'!

initialize

	super initialize.
	connAssocs := Dictionary new.
	compAssocs := Dictionary new.
	number := 1.
	^self! !

!TRDiagnosisSystemWithAssociations methodsFor: 'adding-removing'!

at: anIdentifier addComponent: aComponent

	(compAssocs includesKey: anIdentifier)
	ifFalse:[ compAssocs at: anIdentifier put: (Set new). ].
	self addComponent: aComponent.
	^(compAssocs at: anIdentifier) add: aComponent!

at: anIdentifier addConnection: aConnection

	(connAssocs includesKey: anIdentifier)
	ifFalse:[ connAssocs at: anIdentifier put: (Set new). ].
	self addConnection: aConnection.
	^(connAssocs at: anIdentifier) add: aConnection!

at: anIdentifier addInputConnection: aConnection

	(connAssocs includesKey: anIdentifier)
	ifFalse:[ connAssocs at: anIdentifier put: (Set new). ].
	self addInput: aConnection.
	^(connAssocs at: anIdentifier) add: aConnection!

at: anIdentifier addOutputConnection: aConnection

	(connAssocs includesKey: anIdentifier)
	ifFalse:[ connAssocs at: anIdentifier put: (Set new). ].
	self addOutput: aConnection.
	^(connAssocs at: anIdentifier) add: aConnection!

nextNumber

	^number := number + 1! !

!TRDiagnosisSystemWithAssociations methodsFor: 'accessing'!

compAssocs

	^compAssocs!

connAssocs

	^connAssocs!

originalComponent: aComponent
	"Answers the component of the original system associated with the given one."

	compAssocs keysAndValuesDo:[ :comp :coll |
		coll do:[ :el | (el = aComponent) ifTrue:[ ^comp ].].].
	^nil!

originalConnection: aConnection
	"Answers the connection of the original system associated with the given one."

	connAssocs keysAndValuesDo:[ :conn :coll |
		coll do:[ :el | (el = aConnection) ifTrue:[ ^conn ].].].
	^nil! !

!TRDiagnosisSystemWithAssociations methodsFor: 'private'!

loadRelatedObservations: aDictionary
	"Load observations related to my connections into me."

	aDictionary keysAndValuesDo:[ :conn :val |
		(connAssocs at: conn ifAbsent:[ OrderedCollection new ]) do:[ :relConn |
			self observation: val on: relConn.].].! !

!TRDiagnosisSystemWithAssociations methodsFor: 'diagnosis'!

originalDiagnoses
	"Converts the previously computed diagnoses to diagnoses for the original system."

	| oriDiags newDiag |
	oriDiags := DiagnoseSet new.
	diagnoses isNil
	ifFalse:[
		diagnoses do:[ :diag |
			newDiag := Diagnose new.
			diag do:[ :el |
				newDiag add: (self originalComponent: el ) ].
			oriDiags add: newDiag ].].
	^oriDiags! !

DiagnosisObject subclass: #TreeRelatedComponent
	instanceVariableNames: 'inputs output modes diagnosisSystem inputVectors '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TreeRelatedComponent comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements a diagnosis component. Every component is assumed to have one
output and a fixed number of inputs which must be determined at object creation time.

Instance variables:

inputs		A dictionary associating port names to connection objects
output		A connection connected with the output port
modes		A set of possible modes (This variable is currently not used for diagnosis)
diagnosisSystem		A link to the diagnosis system where self is used.
inputVectors			Stores the input vectors for expected outputs. The stored information
					is used by the TREE algorithm.

The output port can be accessed the same way as the input ports. The name of the output
port is fixed to #out.'!


!TreeRelatedComponent methodsFor: 'testing'!

existsPort: aPortId
	"Answers true if aPortId denotes a port"

	^aPortId == #out or:[ inputs includesKey: aPortId ].! !

!TreeRelatedComponent methodsFor: 'adding-removing'!

connect: aConnection usingPort: aPortId

	(aPortId == #out)
	ifTrue:[
		output := aConnection.
		aConnection from: self. ]
	ifFalse:[
		(self existsPort: aPortId)
		ifTrue:[
			inputs at: aPortId put: aConnection.
			aConnection addPort: (TreeRelatedPort identifier: aPortId component: self). ]
		ifFalse:[ self error: 'Port ', (aPortId printString), ' does not exist!! '.].].! !

!TreeRelatedComponent methodsFor: 'accessing'!

connection: aPortId
	"Answers the connection connected via the given port."

	(aPortId == #out)
	ifTrue:[ ^output ]
	ifFalse:[ ^inputs at: aPortId ifAbsent:[ nil ].].!

diagnosisSystem

	^diagnosisSystem!

diagnosisSystem: aDiagSystem

	^diagnosisSystem := aDiagSystem!

inputs

	^inputs!

modes

	^modes!

nabMode

	^#nab!

output

	^output!

shortName

	^identifier! !

!TreeRelatedComponent methodsFor: 'initialize-release'!

createInputVectors
	"This method should be overwritten by my subclasses to implement the input
	function."

	^Dictionary new!

createPorts: aCollection
	"Creates input ports..."

	inputs := Dictionary new.
	aCollection do:[ :el |
		inputs at: el put: nil ].
	^self!

defaultModes
	"Returns a collection with the default modes."

	^OrderedCollection with: #nab with: #ab!

defaultPorts

	^#(#in1 #in2)!

initialize

	super initialize.
	inputs := Dictionary new.
	output := nil.
	modes := self defaultModes.
	self createPorts: (self defaultPorts).
	inputVectors := self createInputVectors.
	^self! !

!TreeRelatedComponent methodsFor: 'diagnosis'!

diagTopDown: aValue
	"Implements the diagnosis part of the TREE algorithm"

	| newDiags collectedDiags |
	newDiags := DiagnoseSet new.
	newDiags add: (Diagnose with: self).
	(self inputFunction: aValue) do:[ :valDict | 
		collectedDiags := DiagnoseSet with: (Diagnose new).
		(self defaultPorts) do:[ :pid |
			((inputs at: pid) value = (valDict at: pid))
			ifFalse:[
				collectedDiags := (
					collectedDiags times: 
						((inputs at: pid) diagTopDown: (valDict at: pid))) 
					reduceDiagnosisSize: (diagnosisSystem maxDiagnosisSize).].].
		newDiags := newDiags union: collectedDiags.].
	^newDiags!

evaluateTopDown
	"Implements the evaluation function"

	inputs do:[ :conn | conn evaluateTopDown ].
	^self function: inputs.!

forwardCheck: aValue for: aPortId
	"Assumes that aValue is my computed output value and propagate it through
	the rest of the system. True is returned if aValue does not contradict an observation
	and false otherwise."

	| dict |
	dict := Dictionary new.
	self defaultPorts do:[ :pid | 
		dict at: pid put: ((inputs at: pid) value)].
	dict at: aPortId put: aValue.
	^output forwardCheck: (self function: dict)!

function: anInputVector
	"Implements my functionality. This method should be overwritten by
	my subclasses."

	^self subclassResponsibility!

inputFunction: aValue
	"Implements the input function used by the tree algorithm.
	A set of input vectors, i.e., array is returned leading to aValue.
	This method can be overwritten by one of my subclasses."

	^inputVectors at: aValue ifAbsent:[ nil ]!

modifiedDiagTopDown: aValue
	"Implements the diagnosis part of the TREE algorithm"

	| newDiags collectedDiags |
	newDiags := DiagnoseSet new.
	newDiags add: (Diagnose with: self).
	(self inputFunction: aValue) do:[ :valDict | 
		collectedDiags := DiagnoseSet with: (Diagnose new).
		(self defaultPorts) do:[ :pid |
			((inputs at: pid) value = (valDict at: pid))
			ifFalse:[
				collectedDiags := (
					collectedDiags times: 
						((inputs at: pid) modifiedDiagTopDown: (valDict at: pid))) 
					reduceDiagnosisSize: (diagnosisSystem maxDiagnosisSize).].].
		newDiags := newDiags union: collectedDiags.].
	^newDiags!

modifiedTreeDiagTopDown
	"Implements the value computation part of the TREE algorithm"

	inputs do:[ :conn | conn modifiedTreeDiagTopDown ].
	^self function: inputs.!

resetValues!

treeDiagTopDown
	"Implements the value computation part of the TREE algorithm"

	inputs do:[ :conn | conn treeDiagTopDown ].
	^self function: inputs.! !

!TreeRelatedComponent methodsFor: 'printing'!

printOn: aStream

	aStream nextPutAll: (identifier printString), ' '.! !

!TreeRelatedComponent methodsFor: 'converting'!

convertToTree: aDiagnosisSystem with: aNewConnection
	"Converts my connected system in a tree."

	| comp |
	comp := self class identifier: ('C', (aDiagnosisSystem nextNumber printString)).
	aDiagnosisSystem at: self addComponent: comp.
	comp connect: aNewConnection usingPort: #out.
	(self defaultPorts) do:[ :pid |
		comp connect: ((inputs at: pid) convertToTree: aDiagnosisSystem) usingPort: pid ].
	^comp! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

TreeRelatedComponent class
	instanceVariableNames: ''!


!TreeRelatedComponent class methodsFor: 'instance-creation'!

identifier: anObject with: anArgument

	^self identifier: anObject! !

TreeRelatedComponent subclass: #TRCXnor
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TRCXnor comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements a XNOR gate (i.e. the eqivalence function).'!


!TRCXnor methodsFor: 'initialize-release'!

createInputVectors

	| dict valDict |
	dict := Dictionary new.
	dict at: true put: (OrderedCollection new).
	dict at: false put: (OrderedCollection new).
	valDict := Dictionary new.
	valDict at: #in1 put: false.
	valDict at: #in2 put: false.
	(dict at: true) add: valDict.
	valDict := Dictionary new.
	valDict at: #in1 put: true.
	valDict at: #in2 put: true.
	(dict at: true) add: valDict.
	valDict := Dictionary new.
	valDict at: #in1 put: false.
	valDict at: #in2 put: true.
	(dict at: false) add: valDict.
	valDict := Dictionary new.
	valDict at: #in1 put: true.
	valDict at: #in2 put: false.
	(dict at: false) add: valDict.
	^dict! !

!TRCXnor methodsFor: 'diagnosis'!

function: anInputVector

	| v1 v2 |
	v1 := ((anInputVector at: #in1) value).
	v2 := ((anInputVector at: #in2) value).
	^(v1 and: [ v2 ]) or: [ ((v1 not) and: [ v2 not ]) ]! !

TreeRelatedComponent subclass: #TRCNor
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TRCNor comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements a NOR gate.'!


!TRCNor methodsFor: 'initialize-release'!

createInputVectors

	| dict valDict |
	dict := Dictionary new.
	dict at: true put: (OrderedCollection new).
	dict at: false put: (OrderedCollection new).
	valDict := Dictionary new.
	valDict at: #in1 put: false.
	valDict at: #in2 put: false.
	(dict at: true) add: valDict.
	valDict := Dictionary new.
	valDict at: #in1 put: false.
	valDict at: #in2 put: true.
	(dict at: false) add: valDict.
	valDict := Dictionary new.
	valDict at: #in1 put: true.
	valDict at: #in2 put: false.
	(dict at: false) add: valDict.
	^dict! !

!TRCNor methodsFor: 'diagnosis'!

function: anInputVector

	^(((anInputVector at: #in1) value) or:[ ((anInputVector at: #in2) value) ]) not! !

TreeRelatedComponent subclass: #TRCXor
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TRCXor comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements a XOR gate.'!


!TRCXor methodsFor: 'diagnosis'!

function: anInputVector

	| v1 v2 |
	v1 := ((anInputVector at: #in1) value).
	v2 := ((anInputVector at: #in2) value).
	^(v1 and: [ v2 not ]) or: [ (v2 and: [ v1 not ]) ]! !

!TRCXor methodsFor: 'initialize-release'!

createInputVectors

	| dict valDict |
	dict := Dictionary new.
	dict at: true put: (OrderedCollection new).
	dict at: false put: (OrderedCollection new).
	valDict := Dictionary new.
	valDict at: #in1 put: false.
	valDict at: #in2 put: false.
	(dict at: false) add: valDict.
	valDict := Dictionary new.
	valDict at: #in1 put: true.
	valDict at: #in2 put: true.
	(dict at: false) add: valDict.
	valDict := Dictionary new.
	valDict at: #in1 put: false.
	valDict at: #in2 put: true.
	(dict at: true) add: valDict.
	valDict := Dictionary new.
	valDict at: #in1 put: true.
	valDict at: #in2 put: false.
	(dict at: true) add: valDict.
	^dict! !

TreeRelatedComponent subclass: #TRCNand
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TRCNand comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements a NAND gate.'!


!TRCNand methodsFor: 'diagnosis'!

function: anInputVector

	^(((anInputVector at: #in1) value) and:[ ((anInputVector at: #in2) value) ]) not! !

!TRCNand methodsFor: 'initialize-release'!

createInputVectors

	| dict valDict |
	dict := Dictionary new.
	dict at: true put: (OrderedCollection new).
	dict at: false put: (OrderedCollection new).
	valDict := Dictionary new.
	valDict at: #in1 put: true.
	valDict at: #in2 put: true.
	(dict at: false) add: valDict.
	valDict := Dictionary new.
	valDict at: #in1 put: false.
	valDict at: #in2 put: true.
	(dict at: true) add: valDict.
	valDict := Dictionary new.
	valDict at: #in1 put: true.
	valDict at: #in2 put: false.
	(dict at: true) add: valDict.
	^dict! !

TreeRelatedComponent subclass: #TRCNot
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TRCNot comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements an inverter.'!


!TRCNot methodsFor: 'initialize-release'!

createInputVectors

	| dict valDict |
	dict := Dictionary new.
	dict at: true put: (OrderedCollection new).
	dict at: false put: (OrderedCollection new).
	valDict := Dictionary new.
	valDict at: #in put: true.
	(dict at: false) add: valDict.
	valDict := Dictionary new.
	valDict at: #in put: false.
	(dict at: true) add: valDict.
	^dict!

defaultPorts

	^#(#in)! !

!TRCNot methodsFor: 'diagnosis'!

function: anInputVector

	^(anInputVector at: #in) value not! !

TreeRelatedComponent subclass: #TRCAnd
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TRCAnd comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements an AND gate.'!


!TRCAnd methodsFor: 'diagnosis'!

function: anInputVector

	^((anInputVector at: #in1) value) and:[ ((anInputVector at: #in2) value) ]! !

!TRCAnd methodsFor: 'initialize-release'!

createInputVectors

	| dict valDict |
	dict := Dictionary new.
	dict at: true put: (OrderedCollection new).
	dict at: false put: (OrderedCollection new).
	valDict := Dictionary new.
	valDict at: #in1 put: true.
	valDict at: #in2 put: true.
	(dict at: true) add: valDict.
	valDict := Dictionary new.
	valDict at: #in1 put: false.
	valDict at: #in2 put: true.
	(dict at: false) add: valDict.
	valDict := Dictionary new.
	valDict at: #in1 put: true.
	valDict at: #in2 put: false.
	(dict at: false) add: valDict.
	^dict! !

TreeRelatedComponent subclass: #TRCGenericComponent
	instanceVariableNames: 'defaultPorts '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TRCGenericComponent comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements a generic diagnosis component where the number of inputs can
be dynamically set during object creation.

Instance variables:

defaultPorts		A set storing the names of the input ports.

Subclass instances of self are created by using the identifier:with: method. The input
ports are created using the names #in1 ... #in<argument>.'!


!TRCGenericComponent methodsFor: 'initialize-release'!

createInputVectors
	"Generic Components should not use this method."

	^self shouldNotImplement!

createInputVectors: anArgument
	"This method should be overwritten by my subclasses to implement the input
	function."

	^Dictionary new!

defaultPorts

	^defaultPorts!

defaultPorts: anArgument

	| array |
	array := Array new: anArgument.
	1 to: anArgument do:[ :index |
		array at: index put: (('in', (index printString)) asSymbol).].
	defaultPorts := array.
	^array!

initialize

	^self initialize: 2!

initialize: anArgument

	identifier := nil.
	inputs := Dictionary new.
	output := nil.
	modes := self defaultModes.
	self createPorts: (self defaultPorts: anArgument).
	inputVectors := self createInputVectors: anArgument.
	^self! !

!TRCGenericComponent methodsFor: 'converting'!

convertToTree: aDiagnosisSystem with: aNewConnection
	"Converts my connected system in a tree."

	| comp |
	comp := self class identifier: ('C', (aDiagnosisSystem nextNumber printString)) with: (inputs size).
	aDiagnosisSystem at: self addComponent: comp.
	comp connect: aNewConnection usingPort: #out.
	(self defaultPorts) do:[ :pid |
		comp connect: ((inputs at: pid) convertToTree: aDiagnosisSystem) usingPort: pid ].
	^comp! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

TRCGenericComponent class
	instanceVariableNames: ''!


!TRCGenericComponent class methodsFor: 'instance-creation'!

identifier: anObject with: anArgument

	| obj |
	obj := super new initialize: anArgument.
	obj identifier: anObject.
	^obj! !

TRCGenericComponent subclass: #TRCGenericAnd
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TRCGenericAnd comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements an generic AND gate.'!


!TRCGenericAnd methodsFor: 'initialize-release'!

createInputVectors: anArgument
	"This method should be overwritten by my subclasses to implement the input
	function."

	| dict valDict |
	dict := Dictionary new.
	dict at: true put: (OrderedCollection new).
	dict at: false put: (OrderedCollection new).

	valDict := Dictionary new.
	1 to: anArgument do:[ :index1 |
		valDict at: (('in', (index1 printString)) asSymbol) put: true.].
	(dict at: true) add: valDict.

	1 to: anArgument do:[ :index2 |
		valDict := Dictionary new.
		1 to: anArgument do:[ :index3 |
			(index3 = index2)
			ifTrue:[ valDict at: (('in', (index3 printString)) asSymbol) put: false.]
			ifFalse:[ valDict at: (('in', (index3 printString)) asSymbol) put: true.].].
		(dict at: false) add: valDict.].
	^dict! !

!TRCGenericAnd methodsFor: 'diagnosis'!

function: anInputVector

	1 to: (inputs size) do:[ :index |
		((anInputVector at: (('in', (index printString)) asSymbol)) value)
		ifFalse:[ ^false ].].
	^true! !

TRCGenericComponent subclass: #TRCGenericNand
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TRCGenericNand comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements a generic NAND gate.'!


!TRCGenericNand methodsFor: 'diagnosis'!

function: anInputVector

	1 to: (inputs size) do:[ :index |
		((anInputVector at: (('in', (index printString)) asSymbol)) value)
		ifFalse:[ ^true ].].
	^false! !

!TRCGenericNand methodsFor: 'initialize-release'!

createInputVectors: anArgument
	"This method should be overwritten by my subclasses to implement the input
	function."

	| dict valDict |
	dict := Dictionary new.
	dict at: true put: (OrderedCollection new).
	dict at: false put: (OrderedCollection new).

	valDict := Dictionary new.
	1 to: anArgument do:[ :index1 |
		valDict at: (('in', (index1 printString)) asSymbol) put: true.].
	(dict at: false) add: valDict.

	1 to: anArgument do:[ :index2 |
		valDict := Dictionary new.
		1 to: anArgument do:[ :index3 |
			(index3 = index2)
			ifTrue:[ valDict at: (('in', (index3 printString)) asSymbol) put: false.]
			ifFalse:[ valDict at: (('in', (index3 printString)) asSymbol) put: true.].].
		(dict at: true) add: valDict.].
	^dict! !

TRCGenericComponent subclass: #TRCGenericOr
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TRCGenericOr comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements an generic OR gate.'!


!TRCGenericOr methodsFor: 'diagnosis'!

function: anInputVector

	1 to: (inputs size) do:[ :index |
		((anInputVector at: (('in', (index printString)) asSymbol)) value)
		ifTrue:[ ^true ].].
	^false! !

!TRCGenericOr methodsFor: 'initialize-release'!

createInputVectors: anArgument
	"This method should be overwritten by my subclasses to implement the input
	function."

	| dict valDict |
	dict := Dictionary new.
	dict at: true put: (OrderedCollection new).
	dict at: false put: (OrderedCollection new).

	valDict := Dictionary new.
	1 to: anArgument do:[ :index1 |
		valDict at: (('in', (index1 printString)) asSymbol) put: false.].
	(dict at: false) add: valDict.

	1 to: anArgument do:[ :index2 |
		valDict := Dictionary new.
		1 to: anArgument do:[ :index3 |
			(index3 = index2)
			ifTrue:[ valDict at: (('in', (index3 printString)) asSymbol) put: true.]
			ifFalse:[ valDict at: (('in', (index3 printString)) asSymbol) put: false.].].
		(dict at: true) add: valDict.].
	^dict! !

TreeRelatedComponent subclass: #TRCIdentity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TRCIdentity comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements a buffer.'!


!TRCIdentity methodsFor: 'diagnosis'!

function: anInputVector

	^(anInputVector at: #in) value! !

!TRCIdentity methodsFor: 'initialize-release'!

createInputVectors

	| dict valDict |
	dict := Dictionary new.
	dict at: true put: (OrderedCollection new).
	dict at: false put: (OrderedCollection new).
	valDict := Dictionary new.
	valDict at: #in put: true.
	(dict at: true) add: valDict.
	valDict := Dictionary new.
	valDict at: #in put: false.
	(dict at: false) add: valDict.
	^dict!

defaultPorts

	^#(#in)! !

TreeRelatedComponent subclass: #TRCOr
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TRCOr comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements an OR gate.'!


!TRCOr methodsFor: 'initialize-release'!

createInputVectors

	| dict valDict |
	dict := Dictionary new.
	dict at: true put: (OrderedCollection new).
	dict at: false put: (OrderedCollection new).
	valDict := Dictionary new.
	valDict at: #in1 put: false.
	valDict at: #in2 put: false.
	(dict at: false) add: valDict.
	valDict := Dictionary new.
	valDict at: #in1 put: false.
	valDict at: #in2 put: true.
	(dict at: true) add: valDict.
	valDict := Dictionary new.
	valDict at: #in1 put: true.
	valDict at: #in2 put: false.
	(dict at: true) add: valDict.
	^dict! !

!TRCOr methodsFor: 'diagnosis'!

function: anInputVector

	^((anInputVector at: #in1) value) or:[ ((anInputVector at: #in2) value) ]! !

TRCGenericComponent subclass: #TRCGenericNor
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TRCGenericNor comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements a generic NOR gate.'!


!TRCGenericNor methodsFor: 'initialize-release'!

createInputVectors: anArgument
	"This method should be overwritten by my subclasses to implement the input
	function."

	| dict valDict |
	dict := Dictionary new.
	dict at: true put: (OrderedCollection new).
	dict at: false put: (OrderedCollection new).

	valDict := Dictionary new.
	1 to: anArgument do:[ :index1 |
		valDict at: (('in', (index1 printString)) asSymbol) put: false.].
	(dict at: true) add: valDict.

	1 to: anArgument do:[ :index2 |
		valDict := Dictionary new.
		1 to: anArgument do:[ :index3 |
			(index3 = index2)
			ifTrue:[ valDict at: (('in', (index3 printString)) asSymbol) put: true.]
			ifFalse:[ valDict at: (('in', (index3 printString)) asSymbol) put: false.].].
		(dict at: false) add: valDict.].
	^dict! !

!TRCGenericNor methodsFor: 'diagnosis'!

function: anInputVector

	1 to: (inputs size) do:[ :index |
		((anInputVector at: (('in', (index printString)) asSymbol)) value)
		ifTrue:[ ^false ].].
	^true! !

DiagnosisObject subclass: #TreeRelatedPort
	instanceVariableNames: 'component '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Tree-Related'!
TreeRelatedPort comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements a port.

Instance variables:

identifier		Stores the port name.
component	Stores the component. '!


!TreeRelatedPort methodsFor: 'accessing'!

component

	^component!

component: aComponent

	^component := aComponent! !

!TreeRelatedPort methodsFor: 'printing'!

printOn: aStream

	aStream nextPutAll: '#( '.
	identifier printOn: aStream.
	aStream nextPutAll: ' '.
	component printOn: aStream.
	aStream nextPutAll: ') '.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

TreeRelatedPort class
	instanceVariableNames: ''!


!TreeRelatedPort class methodsFor: 'instance-creation'!

identifier: aPortId component: aComponent

	| port |
	port := self new.
	port identifier: aPortId.
	port component: aComponent.
	^port! !

