Object subclass: #DiagnosisObject
	instanceVariableNames: 'identifier '
	classVariableNames: 'DiagnosisErrorSignal '
	poolDictionaries: ''
	category: 'Diagnosis-Systems'!


!DiagnosisObject methodsFor: 'initialize-release'!

initialize

	identifier := nil.
	^self! !

!DiagnosisObject methodsFor: 'accessing'!

identifier

	^identifier!

identifier: anObject

	^identifier := anObject!

named

	^identifier! !

!DiagnosisObject methodsFor: 'error-handling'!

error: aText
	"Raises the DiagnosisErrorSignal with the given message."

	^self class error: aText! !

!DiagnosisObject methodsFor: 'printing'!

insertJunk: aNumber on: aStream

	(aNumber > 100)
	ifTrue:[
		aStream nextPutAll: '!!
'.
		^1 ]
	ifFalse:[
		aStream nextPutAll: '
'.
		^aNumber + 1 ].! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

DiagnosisObject class
	instanceVariableNames: ''!


!DiagnosisObject class methodsFor: 'instance-creation'!

identifier: anObject

	| obj |
	obj := super new initialize.
	obj identifier: anObject.
	^obj!

new

	^super new initialize! !

!DiagnosisObject class methodsFor: 'class-initialization'!

initialize
	"DiagnosisObject initialize"

	DiagnosisErrorSignal := Signal new.
	^self! !

!DiagnosisObject class methodsFor: 'accessing'!

diagnosisErrorSignal

	^DiagnosisErrorSignal! !

!DiagnosisObject class methodsFor: 'error-handling'!

error: aText
	"Raises the DiagnosisErrorSignal with the given message."

	DiagnosisObject diagnosisErrorSignal raiseErrorString: aText! !

DiagnosisObject subclass: #DiagnosisConnection
	instanceVariableNames: 'diagnosisSystem observable observationCost '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Systems'!
DiagnosisConnection comment:
'This class implements the generic diagnosis connection.'!


!DiagnosisConnection methodsFor: 'printing'!

name

	^identifier printString!

printOn: aStream
	"Writes self on aStream."

	identifier printOn: aStream.!

shortName

	^identifier printString! !

!DiagnosisConnection methodsFor: 'diagnosis'!

setObservation: aValue
	"Sets self to the given observed value."

	^self subclassResponsibility.! !

!DiagnosisConnection methodsFor: 'accessing'!

allConnectedComponents
	"Answers a set of components which are connected with me."

	self subclassResponsibility!

diagnosisSystem

	^diagnosisSystem!

diagnosisSystem: aSystem

	^diagnosisSystem := aSystem!

named

	^identifier printString!

observationCost

	^observationCost!

observationCost: aValue

	^observationCost := aValue!

setToNotObservable

	^observable := false.!

setToObservable

	^observable := true.! !

!DiagnosisConnection methodsFor: 'testing'!

isConnected
	"Answers true if self is connected."

	self subclassResponsibility!

isInput

	self subclassResponsibility!

isObservable

	^observable!

isOutput

	self subclassResponsibility!

storesMultipleValues

	^false! !

!DiagnosisConnection methodsFor: 'initialize-release'!

initialize

	identifier := nil.
	observable := true.
	observationCost := 0.0d.
	^self! !

DiagnosisConnection subclass: #CPDiagnosisConnection
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Systems'!


!CPDiagnosisConnection methodsFor: 'diagnosis'!

allValues

	^identifier allValues!

setObservation: aValue
	"Sets self to the given observed value."

	(self storesMultipleValues)
	ifTrue:[ aValue do:[ :val | identifier setValue: val ].]
	ifFalse:[ identifier setValue: aValue. ].!

value
	"Returns my current value."

	^identifier value! !

!CPDiagnosisConnection methodsFor: 'accessing'!

allConnectedComponents
	"Answers a set of components which are connected with me."

	| cons result |
	cons := identifier allConnectedConstraints.
	result := Set new.
	diagnosisSystem diagnosisComponents do:[ :comp |
		(cons includes: (comp identifier))
		ifTrue:[ result add: comp ].].
	^result!

named

	^identifier identifier! !

!CPDiagnosisConnection methodsFor: 'printing'!

name

	^identifier identifier printString! !

!CPDiagnosisConnection methodsFor: 'testing'!

isConnected
	"Answers true if self is connected."

	^identifier isConnected!

isInput

	^identifier isInput!

isOutput

	^identifier isOutput!

storesMultipleValues

	^identifier storesMultipleValues! !

DiagnosisObject subclass: #GenericDiagnosisSystem
	instanceVariableNames: 'components connections observations maxNumberOfDiagnoses maxDiagnosisSize diagnoses '
	classVariableNames: 'ActualSystem '
	poolDictionaries: ''
	category: 'Diagnosis-Systems'!
GenericDiagnosisSystem comment:
'(C) 1997 F. Wotawa, Technische Universitaet Wien
============================================================================
This class implements a generic diagnosis system. Only instance variables and their methods
are defined. Functionality must be implemented by my subclasses.'!


!GenericDiagnosisSystem methodsFor: 'accessing'!

components

	^components!

connections

	^connections!

diagnoses

	^diagnoses!

maxDiagnosisSize

	^maxDiagnosisSize!

maxDiagnosisSize: aNumber

	^maxDiagnosisSize := aNumber!

maxNumberOfDiagnoses

	^maxNumberOfDiagnoses!

maxNumberOfDiagnoses: aNumber

	^maxNumberOfDiagnoses := aNumber!

observations
	"Answers a dictionary containing connections as key and their observed value
	as associated value field."

	^observations! !

!GenericDiagnosisSystem methodsFor: 'initialize-release'!

initialize

	super initialize.
	components := Set new.
	connections := Set new.
	observations := Dictionary new.
	maxNumberOfDiagnoses := 1000.
	maxDiagnosisSize := 1.
	^self! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GenericDiagnosisSystem class
	instanceVariableNames: ''!


!GenericDiagnosisSystem class methodsFor: 'accessing'!

actualSystem

	^ActualSystem!

actualSystem: aDiagnosisSystem

	^ActualSystem := aDiagnosisSystem! !

!GenericDiagnosisSystem class methodsFor: 'class-initialization'!

initialize
	"GenericDiagnosisSystem initialize"

	ActualSystem := nil.
	^self! !

DiagnosisObject subclass: #DiagnosisFMEA
	instanceVariableNames: 'observations components fmeaCollection diagnosisSystem '
	classVariableNames: 'ActualFMEA '
	poolDictionaries: ''
	category: 'Diagnosis-Systems'!


!DiagnosisFMEA methodsFor: 'accessing'!

addComponent: aDiagComp

	^components add: aDiagComp!

components

	^components!

components: aCollection

	^components := aCollection!

diagnosisSystem

	^diagnosisSystem!

diagnosisSystem: aDiagSystem

	^diagnosisSystem := aDiagSystem!

fmeaCollection

	^fmeaCollection!

fmeaCollection: anOrderedCollection

	^fmeaCollection := anOrderedCollection!

observation: aValue on: aConnection
	"Adds a new observation to self."

	(aConnection storesMultipleValues)
	ifTrue:[
		(observations at: aConnection ifAbsent:[ nil ]) isNil
		ifTrue:[ observations at: aConnection put: (Set with: aValue) ]
		ifFalse:[ (observations at: aConnection) add: aValue ].]
	ifFalse:[ observations at: aConnection put: aValue ].
	^aValue!

observations

	^observations!

observations: aDictionary

	^observations := aDictionary! !

!DiagnosisFMEA methodsFor: 'fmea-handling'!

addOBS: aDictionary for: aDiagnosis
	"Adds an FMEA element to self."

	| dict |
	dict := Dictionary new.
	aDictionary keysAndValuesDo:[ :conn :val |
		(observations includesKey: conn)
		ifFalse:[
			conn storesMultipleValues
			ifTrue:[
				val isEmpty ifFalse:[ dict at: conn put: val ].]
			ifFalse:[ 
				val isNil ifFalse:[ dict at: conn put: val ].].].].
	^fmeaCollection addLast: (aDiagnosis -> dict).!

allDiagnoses
	"Returns a set of all diagnoses stored in my structure."

	| diags |
	diags := DiagnoseSet new.
	fmeaCollection do:[ :assoc | diags add: (assoc key).].
	^diags!

diagnosesFor: aValue and: aConnection
	"Answers a set of diagnosis for which the observation given by aValue and aConnection
	leads not to an contradiction."

	| diags flag |
	diags := DiagnoseSet new.
	fmeaCollection do:[ :assoc | | diag obs |
		diag := assoc key.
		obs := assoc value at: aConnection ifAbsent:[ nil ].
		obs isNil
		ifTrue:[ diags add: diag. ]
		ifFalse:[
			aConnection storesMultipleValues
			ifTrue:[
				flag := true.
				obs do:[ :val |
					(val contradicts: aValue) ifTrue:[ flag := false ].].
				flag ifTrue:[ diags add: diag. ].]
			ifFalse:[ (aValue = obs) ifTrue:[ diags add: diag. ].].].].
	^diags! !

!DiagnosisFMEA methodsFor: 'initialize-release'!

initialize

	identifier := nil.
	components := Set new.
	fmeaCollection := OrderedCollection new.
	observations := Dictionary new.
	^self! !

!DiagnosisFMEA methodsFor: 'printing'!

printFMEA: aStream
	"Writes my contents to the given stream."

	| lc |
	lc := 1.
	aStream nextPutAll: (self class printString), ' fmeaIdentifier: ', (identifier printString), '.'; cr.
	lc := self insertJunk: lc on: aStream.
	components do:[ :comp |
		aStream nextPutAll: (self class printString), ' component: ', (comp named printString), '.'.
		lc := self insertJunk: lc on: aStream. ].
	aStream cr.
	observations keysAndValuesDo:[ :conn :val |
		conn storesMultipleValues
		ifTrue:[
			val do:[ :el |
				aStream nextPutAll: (self class printString), ' observe: ', (el printString), ' on: ', (conn name), '.'.
				lc := self insertJunk: lc on: aStream.].]
		ifFalse:[
			aStream nextPutAll: (self class printString), ' observe: ', (val printString), ' on: ', (conn name), '.'.
			lc := self insertJunk: lc on: aStream. ].].
	aStream cr.
	fmeaCollection do:[ :assoc | | diag obs |
		diag := assoc key.
		obs := assoc value.
		aStream nextPutAll: (self class printString), ' addOBS: #('.
		obs keysAndValuesDo:[ :conn :val |
			conn storesMultipleValues
			ifTrue:[ val do:[ :el | aStream nextPutAll: ' #( ''', (el printString), ''' ', (conn name), ' ) '.].]
			ifFalse:[ aStream nextPutAll: ' #( ''', (val printString), ''' ', (conn name), ' ) '.]. ].
		aStream nextPutAll: ' ) for: #('.
		diag do:[ :cma |
			aStream
				nextPutAll: '#( ', (cma mode printString), ' ', (cma component name), ' ) '. ].
		aStream nextPutAll: ' ). '.
		lc := self insertJunk: lc on: aStream.].! !

!DiagnosisFMEA methodsFor: 'file-accessing'!

saveFMEA: aFilename
	"Saves my contents into the given file."

	| stream |
	stream := aFilename asFilename writeStream.
	self printFMEA: stream.
	stream close.
	^self! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

DiagnosisFMEA class
	instanceVariableNames: ''!


!DiagnosisFMEA class methodsFor: 'instance-creation'!

diagnosisSystem: aDiagSystem

	| fmea |
	fmea := self new.
	fmea diagnosisSystem: aDiagSystem.
	fmea identifier: 'FMEA for ', (aDiagSystem named).
	fmea components: (aDiagSystem focus copy).
	fmea observations: (aDiagSystem observations copy).
	^fmea! !

!DiagnosisFMEA class methodsFor: 'file-accessing'!

loadFMEA: aFilename using: aDiagSystem
	"Loads a new FMEA stored in aFilename. A new instance of self is created."

	(aFilename asFilename exists)
	ifTrue:[
		DiagnosisFMEA actualFMEA: (self new).
		DiagnosisFMEA actualFMEA diagnosisSystem: aDiagSystem.
		aFilename asFilename fileIn. ]
	ifFalse:[ self error: 'The file ', (aFilename printString), ' does not exist or is not readable' ].
	^DiagnosisFMEA actualFMEA! !

!DiagnosisFMEA class methodsFor: 'accessing'!

actualFMEA

	^ActualFMEA!

actualFMEA: aFMEA

	^ActualFMEA := aFMEA! !

!DiagnosisFMEA class methodsFor: 'file-accessing-private'!

addOBS: anOBSArray for: anDiagArray

	| valDict diag |
	(self actualFMEA isNil)
	ifTrue:[ ^self error: 'No FMEA has been spezified.' ]
	ifFalse:[ 
		(self actualFMEA diagnosisSystem isNil)
		ifTrue:[ ^self error: 'A diagnosis system must be given before calling the component: method.'. ]
		ifFalse:[
			valDict := self createValueDictionary: anOBSArray.
			diag := self createDiagnosis: anDiagArray.
			self actualFMEA addOBS: valDict for: diag. ].].!

component: anIdentifier

	| comp |
	(self actualFMEA isNil)
	ifTrue:[ ^self error: 'No FMEA has been spezified.' ]
	ifFalse:[ 
		(self actualFMEA diagnosisSystem isNil)
		ifTrue:[ ^self error: 'A diagnosis system must be given before calling the component: method.'. ]
		ifFalse:[
			comp := self actualFMEA diagnosisSystem componentNamed: anIdentifier.
			comp isNil
			ifTrue:[ ^self error: 'The component ', (anIdentifier printString), ' does not exist.'. ]
			ifFalse:[ self actualFMEA addComponent: comp. ].].].!

createDiagnosis: aDiagArray

	| diag |
	diag := ExtendedDiagnosis new.
	aDiagArray do:[ :cmaArray | | comp |
		comp := self actualFMEA diagnosisSystem componentNamed: (cmaArray at: 2).
		comp isNil
		ifTrue:[ ^self error: 'Component ', ((cmaArray at: 2) printString), ' does not exist.' ]
		ifFalse:[ diag addComponent: comp mode: (cmaArray at: 1).].].
	^diag!

createValueDictionary: aValArray

	| dict |
	dict := Dictionary new.
	aValArray do:[ :valElement | | conn val |
		conn := self actualFMEA diagnosisSystem connectionNamed: (valElement at: 2).
		val := Compiler evaluate: (valElement at: 1).
		conn isNil
		ifTrue:[ ^self error: 'Connection ', ((valElement at: 2) printString), ' does not exist.'. ]
		ifFalse:[
			(conn storesMultipleValues)
			ifTrue:[
				(dict at: conn ifAbsent:[ nil ]) isNil
				ifTrue:[ dict at: conn put: (Set with: val) ]
				ifFalse:[ (dict at: conn) add: val ].]
			ifFalse:[ dict at: conn put: val ].].].
	^dict!

fmeaIdentifier: anIdentifier

	(self actualFMEA isNil)
	ifTrue:[ ^self error: 'No FMEA has been spezified.' ]
	ifFalse:[ self actualFMEA identifier: anIdentifier. ].!

observe: aValue on: anIdentifier

	| conn |
	(self actualFMEA isNil)
	ifTrue:[ ^self error: 'No FMEA has been spezified.' ]
	ifFalse:[ 
		(self actualFMEA diagnosisSystem isNil)
		ifTrue:[ ^self error: 'A diagnosis system must be given before calling the component: method.'. ]
		ifFalse:[
			conn := self actualFMEA diagnosisSystem connectionNamed: anIdentifier.
			conn isNil
			ifTrue:[ ^self error: 'The connection ', (anIdentifier printString), ' does not exist.'. ]
			ifFalse:[ self actualFMEA observation: aValue on: conn. ].].].! !

GenericDiagnosisSystem subclass: #DiagnosisSystem
	instanceVariableNames: 'focus maxFMEASize actualDiagnosis theoremProver compNames connNames fmea '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Systems'!


!DiagnosisSystem methodsFor: 'initialize-release'!

defaultTheoremProverClass
	"Answers the default theorem prover class."

	^self class defaultTheoremProverClass!

initialize

	identifier := nil.
	components := Set new.
	connections := Set new.
	observations := Dictionary new.
	focus := Set new.
	maxNumberOfDiagnoses := 1000.
	maxDiagnosisSize := 1.
	maxFMEASize := 1.
	diagnoses := Set new.
	actualDiagnosis := false.
	theoremProver := self defaultTheoremProverClass new.
	compNames := Dictionary new.
	connNames := Dictionary new.
	^self! !

!DiagnosisSystem methodsFor: 'public-accessing-structure'!

component: aCompId type: aType
	"Adds a component named by aCompId of aType to self."

	DiagnosisSystem actualSystem: self.
	self class component: aCompId type: aType.!

connect: aConnId withPort: aPortId at: aCompId
	"Connects a connection with a port of a component."

	DiagnosisSystem actualSystem: self.
	self class connect: aConnId withPort: aPortId at: aCompId.!

connection: aConnId
	"Adds a connection named by aConnId to self."

	DiagnosisSystem actualSystem: self.
	self class connection: aConnId.!

connection: aConnId type: aType
	"Adds a connection named by aConnId of aType to self."

	DiagnosisSystem actualSystem: self.
	self class connection: aConnId type: aType.!

connectionFor: aCompId withPort: aPortId
	"Returns the identifier of the connection connected with the given
	component using the given port."

	^self subclassResponsibility!

deleteComponent: aCompId
	"Removes the component named by aCompId if it is not used."

	| comp |
	comp := self componentNamed: aCompId.
	comp isNil
	ifFalse:[
		comp isConnected
		ifFalse:[ 
			components remove: comp. 
			compNames removeKey: aCompId. ].].!

deleteConnection: aConnId
	"Removes the connection named by aConnId if it is not used."

	| conn |
	conn := self connectionNamed: aConnId.
	conn isNil
	ifFalse:[
		conn isConnected
		ifFalse:[ 
			connections remove: conn. 
			connNames removeKey: aConnId. ].].!

disconnect: aCompId withPort: aPortId
	"Removes the connection from the given port of the given
	component."

	^self subclassResponsibility! !

!DiagnosisSystem methodsFor: 'public-accessing'!

activeFaultMode: aMode for: aComponent
	"Sets the given mode of the component to the active state. Only
	active modes are considered for diagnosis."

	actualDiagnosis := false.
	^aComponent activeFaultMode: aMode!

addToFocus: aComponent
	"Adds the given component to the focus."

	(components includes: aComponent) ifFalse:[ self error: 'Component ', (aComponent printString), ' does not exists!!!!' ].
	actualDiagnosis := false.
	^focus add: aComponent!

componentNamed: anObject
	"Answers the component named by anObject if it exists and nil otherwise."

	^compNames at: anObject ifAbsent:[ nil ]!

connectionNamed: anObject
	"Answers the connection named by anObject if it exists and nil otherwise."

	^connNames at: anObject ifAbsent:[ nil ]!

diagnosisComponents
	"Answers a set of components used for diagnosis."

	^components!

diagnosisConnections
	"Answers a set of connections used for diagnosis."

	^connections!

faultModes: aComponent
	"Answers a set of fault modes defined for the given component."

	^aComponent faultModes!

faultProbability: aValue for: aComponent
	"Sets the fault probability of the abnormal mode of the given component
	to the given value. This value must not be greater or equal 0.5."

	actualDiagnosis := false.
	^aComponent faultProbability: aValue!

faultProbability: aValue for: aMode and: aComponent
	"Sets the fault probability of the given mode of the given component
	to the given value. If the sum of all mode probabilities exceeds 1 no
	further changes can be made."

	actualDiagnosis := false.
	^aComponent faultProbability: aValue for: aMode!

faultProbabilityFor: aComponent
	"Answers the fault probability of the abnormal mode for the given component."

	^aComponent faultProbability!

faultProbabilityFor: aMode and: aComponent
	"Answers the fault probability of the given mode for the given component."

	^aComponent faultProbabilityFor: aMode!

focus
	"Returns the current focus, i.e., a set of diagnosis components."

	^focus!

isActiveFaultMode: aMode for: aComponent
	"Answers true if the given mode of the given component
	is in the active state and false otherwise."

	^aComponent isActiveFaultMode: aMode!

isPassiveFaultMode: aMode for: aComponent
	"Answers true if the given mode of the given component
	is in the passive state and false otherwise."

	^aComponent isPassiveFaultMode: aMode!

observation: aValue on: aConnection
	"Adds a new observation to self."

	actualDiagnosis := false.
	(aConnection storesMultipleValues)
	ifTrue:[
		(observations at: aConnection ifAbsent:[ nil ]) isNil
		ifTrue:[ observations at: aConnection put: (Set with: aValue) ]
		ifFalse:[ (observations at: aConnection) add: aValue ].]
	ifFalse:[ observations at: aConnection put: aValue ].
	^aValue!

observationValue: aConnection
	"Returns the observed value associated with the given connection.
	nil is returned if no such observation exists."

	^observations at: aConnection ifAbsent:[ nil ]!

passiveFaultMode: aMode for: aComponent
	"Sets the given mode of the component to the passive state.
	Passive modes are not considered for diagnosis."

	actualDiagnosis := false.
	^aComponent passiveFaultMode: aMode!

removeFromFocus: aComponent
	"Removes the given component from the focus."

	actualDiagnosis := false.
	^focus remove: aComponent!

removeOBS: aConnection
	"Removes the observation related to the given connection."

	actualDiagnosis := false.
	^observations removeKey: aConnection ifAbsent:[ nil ]!

systemName

	^self named! !

!DiagnosisSystem methodsFor: 'simulation'!

computeOutputValues
	"Computes the value for every output connection assuming the correctness of all
	components. A dictionary containing connections associated with their value
	is returned."

	^self computeOutputValues: (ExtendedDiagnosis new)!

computeOutputValues: aDiagnosis
	"Computes the values of all output connections assuming that the components
	behave like given by aDiagnosis."

	^self subclassResponsibility!

simulate
	"Computes the value for every connection assuming the correctness of all
	components. A dictionary containing connections associated with their value
	is returned."

	^self simulate: (ExtendedDiagnosis new)!

simulate: aDiagnosis
	"Computes the values of all connections assuming that the components
	behave like given by aDiagnosis."

	^self subclassResponsibility! !

!DiagnosisSystem methodsFor: 'testcase-computation'!

computeOBS
	"Computes output observations using the current input observations."

	| inputDict |
	inputDict := Dictionary new.
	self allInputConnections do:[ :id | | val |
		val := observations at: id ifAbsent:[ nil ].
		val isNil ifFalse:[ inputDict at: id put: val ].].
	^self computeOBSusingInputs: inputDict andOutputs: (self allOutputConnections)!

computeOBSForOutputs: anOutputCollection
	"Computes a collection of observations containing an output vector for
	the given observations previously stored in self. The output vector is computed randomly by
	assuming that at least one value is wrong. A dictionary associating a value to
	every element of anOutputCollection is returned as result."

	^Dictionary new!

computeOBSusingInputs: anOBSDictionary andOutputs: anOutputCollection
	"Computes a collection of observations containing an output vector for
	the given input vector related observations. The output vector is computed randomly by
	assuming that at least one value is wrong. A dictionary associating a value to
	every element of anOutputCollection is returned as result."

	| result obsDict |
	obsDict := observations.
	observations := anOBSDictionary.
	result := self computeOBSForOutputs: anOutputCollection.
	observations := obsDict.
	^result! !

!DiagnosisSystem methodsFor: 'diagnosis'!

canBeEntered: aComponent
	"Answers true if the given component can be entered for hierarchic diagnosis."

	^aComponent canBeEntered!

computeDiagnosesProbabilities
	"Computes the probabilities for every diagnosis."

	diagnoses isNil
	ifFalse:[
		diagnoses containsEmptyDiagnosis
		ifTrue:[ diagnoses := DiagnoseSet with: (ExtendedDiagnosisProbability probability: (self probabilityOfEmptyDiagnosis)). ]
		ifFalse:[ diagnoses := diagnoses withProbabilities. ].].!

computeDiscrepancies
	"Computes the current discrepancies."

	| outVals discr label inLabel |
	outVals := self computeOutputValues.
	discr := OrderedCollection new.
	outVals keysAndValuesDo:[ :con :val | | obsVal |
		obsVal := observations at: con ifAbsent:[ nil ].
		obsVal isNil
		ifFalse:[
			con storesMultipleValues
			ifTrue:[
				label := false.
				val do:[ :el |
					inLabel := false.
					obsVal do:[ :obsEl |
						obsEl = el ifTrue:[ inLabel := true ].].
					inLabel ifFalse:[ label := true ].].
				label ifFalse:[ discr add: con ].]
			ifFalse:[
				val = obsVal ifFalse:[ discr add: con ].].].].
	^discr!

enter: aComponent
	"Answers a diagnosis system related to the given component."

	^aComponent enter!

focusOnDiscrepancies
	"Computes a focus using computed output values not equal to their observed values.
	Using makeDiagnosis after calling this method may lead to ingore valid diagnosis."

	| discr |
	discr := self computeDiscrepancies.
	discr isEmpty
	ifTrue:[ focus := components copy ]
	ifFalse:[
		focus := Set new.
		discr do:[ :con | focus addAll: (con allConnectedComponents) ].
		maxDiagnosisSize := (discr size).].!

hasDiagnosis
	"Answers true if diagnoses has been previously computed
	(before changes have been done)."

	^actualDiagnosis!

isConsistent

	^theoremProver isConsistent!

makeDiagnosis
	"Starts the diagnosis procedure. This method should be overwritten by my subclasses."

	^self subclassResponsibility!

makeDiagnosisWithFaultProbabilities
	"Starts the diagnosis procedure. This method should be overwritten by my subclasses."

	^self subclassResponsibility!

makeDiagnosisWithFMEA
	"Computes diagnosis using a previously computed FMEA."

	| diags |
	fmea isNil
	ifTrue:[ ^self error: 'No FMEA has been loaded or created before calling the FMEA diagnosis method.'.].
	diags := fmea allDiagnoses.
	observations keysAndValuesDo:[ :conn :val |
		conn storesMultipleValues
		ifTrue:[ val do:[ :el | diags := diags intersect: (fmea diagnosesFor: el and: conn).].]
		ifFalse:[ diags := diags intersect: (fmea diagnosesFor: val and: conn).].].
	^diags minimize!

makeFMEA
	"Computes a FMEA out of the given observations. The size of the FMEA is
	specified by maxFMEASize."

	| behav |
	behav := self simulate.
	self isConsistent
	ifFalse:[ ^self error: 'A FMEA can not be done because of inconsistent observations.'. ].
	fmea := DiagnosisFMEA diagnosisSystem: self.
	self
		makeFMEA: fmea
		usedComponents: (ExtendedDiagnosis new)
		withComponents: ((ExtendedDiagnosis withOKComponents: focus) asExtendedChoice)
		withBehavior: behav
		actualSize: 1.
	^fmea!

makeFMEA: aFMEA usedComponents: usedComps withComponents: nextComps withBehavior: aValDict actualSize: aNumber
	"Computes a FMEA out of the given observations. Computation is done in a depth-first manner!!"

	| newNext |
	((aNumber > maxFMEASize) or:[ nextComps isEmpty ] ) ifTrue:[ ^aFMEA ].
	newNext := nextComps copy.
	nextComps do:[ :cma | | diag |
		newNext remove: cma.
		diag := usedComps union: (ExtendedDiagnosis with: cma).
		aFMEA addOBS: (self simulate: diag) for: diag.
		self makeFMEA: aFMEA usedComponents: diag withComponents: newNext withBehavior: aValDict actualSize: (aNumber + 1). ].
	^aFMEA!

makeIterativeDiagnosis
	"Starts the diagnosis procedure. This method should be overwritten by my subclasses.
	This method searches for at least one diagnosis."

	^self subclassResponsibility!

measurementSelection
	"Starts the measurement selection procedure. Answers a dictionary containing connections
	and associated value indicating importance. A higher value means higher importance."

	^self subclassResponsibility!

probabilityOfEmptyDiagnosis
	"Answers the probability of the empty diagnosis."

	| prob |
	prob := 1.0d.
	components do:[ :comp |
			prob := prob * (comp faultProbabilityFor: (comp nabMode)).].
	^prob!

resetToDefaults
	"Resets diagnosis information to default values."

	focus := components copy.
	components do:[ :comp | comp resetToDefaults ].
	^self! !

!DiagnosisSystem methodsFor: 'accessing'!

addComponent: aDiagnosisComponent

	compNames at: (aDiagnosisComponent identifier) put: aDiagnosisComponent.
	^components add: aDiagnosisComponent!

addConnection: aDiagnosisConnection

	connNames at: (aDiagnosisConnection identifier) put: aDiagnosisConnection.
	^connections add: aDiagnosisConnection!

allInputConnections
	"Answers a set of input connections."

	^connections select:[ :el | el isInput ].!

allOutputConnections
	"Answers a set of input connections."

	^connections select:[ :el | el isOutput ].!

fmea

	^fmea!

maxFMEASize

	^maxFMEASize!

maxFMEASize: aNumber

	^maxFMEASize := aNumber!

removeDiagnoses
	"Removes all diagnoses from self."

	^diagnoses := DiagnoseSet new.!

removeOBS
	"Removes all observations from self."

	^observations := Dictionary new.!

theoremProver

	^theoremProver!

theoremProver: aTheoremProver

	^theoremProver := aTheoremProver! !

!DiagnosisSystem methodsFor: 'printing'!

printDiagnosesOn: aStream
	"Prints my diagnoses on a stream. This method is used to save the current 
	diagnoses in a file."

	| lc |
	lc := 1.
	diagnoses do:[ :diag |
		aStream
			nextPutAll: (self class printString), ' diagnosis: #( '.
		diag do:[ :cma |
			aStream
				nextPutAll: '#( ', (cma mode printString), ' ', (cma component name), ' ) '. ].
		aStream nextPutAll: ').'.
		lc := self insertJunk: lc on: aStream.].!

printOBSOn: aStream
	"Prints my observations on a stream. This method is used to save observations 
	in a file."

	| lc |
	lc := 1.
	observations keysAndValuesDo:[ :conn :val |
		conn storesMultipleValues
		ifTrue:[
			val do:[ :el |
				aStream nextPutAll: (self class printString), ' observe: ', (el printString), ' on: ', (conn name), '.'.
				lc := self insertJunk: lc on: aStream.].]
		ifFalse:[
			aStream nextPutAll: (self class printString), ' observe: ', (val printString), ' on: ', (conn name), '.'.
			lc := self insertJunk: lc on: aStream. ].].!

printParametersOn: aStream
	"Prints my diagnosis parameters on a stream. This method is used to save parameters 
	in a file."

	| lc |
	aStream
		nextPutAll: (self class printString), ' maxNumberOfDiagnoses: ', (maxNumberOfDiagnoses printString), '.';
		cr.
	aStream
		nextPutAll:  (self class printString), ' maxDiagnosisSize: ', (maxDiagnosisSize printString), '.';
		cr.
	aStream
		nextPutAll: (self class printString), ' maxFMEASize: ', (maxFMEASize printString), '.';
		cr.
	lc := 5.
	components do:[ :dc |
		dc modes do:[ :cm |
			aStream nextPutAll: 
	 (self class printString), ' component: ', (dc named printString), ' faultProbability: ', (dc faultProbabilityValueFor: cm) printString, ' for: ', (cm printString), '.'.
			lc := self insertJunk: lc on: aStream. ].].! !

!DiagnosisSystem methodsFor: 'file-accessing'!

loadDiagnoses: aFilename
	"Loads the diagnoses stored in aFilename. The current diagnosis set is removed."

	DiagnosisSystem actualSystem: self.
	DiagnosisSystem loadDiagnoses: aFilename.
	^self!

loadFMEA: aFilename
	"Loads a FMEA stored in aFilename."

	(aFilename asFilename exists)
	ifTrue:[ ^fmea := DiagnosisFMEA loadFMEA: aFilename using: self ]
	ifFalse:[ ^self error: 'The file ', (aFilename printString), ' does not exist or is not readable' ].!

loadOBS: aFilename
	"Loads the observation stored in aFilename."

	DiagnosisSystem actualSystem: self.
	DiagnosisSystem loadOBS: aFilename.
	^self!

loadParameters: aFilename
	"Loads the parameters stored in aFilename."

	DiagnosisSystem actualSystem: self.
	DiagnosisSystem loadParameters: aFilename.
	^self!

loadSD: aFilename
	"Loads a new system stored in aFilename. A new instance of self is created."

	^self class loadSD: aFilename.!

saveDiagnoses: aFilename
	"Saves the current diagnoses in aFilename"

	| stream |
	stream := aFilename asFilename writeStream.
	self printDiagnosesOn: stream.
	stream close.
	^self!

saveFMEA: aFilename
	"Saves the current diagnosis parameters in aFilename"

	fmea isNil
	ifTrue:[ ^self error: 'No FMEA has been created.'. ]
	ifFalse:[ ^fmea saveFMEA: aFilename.].!

saveOBS: aFilename
	"Saves the current observation in aFilename"

	| stream |
	stream := aFilename asFilename writeStream.
	self printOBSOn: stream.
	stream close.
	^self!

saveParameters: aFilename
	"Saves the current diagnosis parameters in aFilename"

	| stream |
	stream := aFilename asFilename writeStream.
	self printParametersOn: stream.
	stream close.
	^self! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

DiagnosisSystem class
	instanceVariableNames: ''!


!DiagnosisSystem class methodsFor: 'file-accessing'!

loadDiagnoses: aFilename
	"Loads diagnoses stored in aFilename into my actual instance."

	(aFilename asFilename exists)
	ifTrue:[
		ActualSystem isNil
		ifTrue:[ self error: 'An actual system must exist before assigning diagnoses to it.' ]
		ifFalse:[
			ActualSystem removeDiagnoses. 
			aFilename asFilename fileIn. ].]
	ifFalse:[ self error: 'The file ', (aFilename printString), ' does not exist or is not readable' ].!

loadOBS: aFilename
	"Loads observations stored in aFilename into my actual instance."

	(aFilename asFilename exists)
	ifTrue:[
		ActualSystem isNil
		ifTrue:[ self error: 'An actual system must exist before assigning observations to it.' ]
		ifFalse:[
			ActualSystem removeOBS. 
			aFilename asFilename fileIn. ].]
	ifFalse:[ self error: 'The file ', (aFilename printString), ' does not exist or is not readable' ].!

loadParameters: aFilename
	"Loads parameters stored in aFilename into my actual instance."

	(aFilename asFilename exists)
	ifTrue:[
		ActualSystem isNil
		ifTrue:[ self error: 'An actual system must exist before assigning parameters to it.' ]
		ifFalse:[
			aFilename asFilename fileIn. ].]
	ifFalse:[ self error: 'The file ', (aFilename printString), ' does not exist or is not readable' ].!

loadSD: aFilename
	"Loads a new system stored in aFilename. A new instance of self is created."

	(aFilename asFilename exists)
	ifTrue:[
		DiagnosisSystem actualSystem: (self new).
		DiagnosisSystem actualSystem theoremProver: (self defaultTheoremProverClass new).
		aFilename asFilename fileIn. ]
	ifFalse:[ self error: 'The file ', (aFilename printString), ' does not exist or is not readable' ].
	^DiagnosisSystem actualSystem! !

!DiagnosisSystem class methodsFor: 'file-accessing-private'!

activeFaultMode: aMode for: anIdentifier
	"Sets the given mode of the given component named by anIdentifier to the
	active state.."

	| comp |
	ActualSystem isNil
	ifTrue:[ ^self error: 'An actual system must exist before searching for a component.' ]
	ifFalse:[ 
		comp := ActualSystem componentNamed: anIdentifier.
		comp isNil ifTrue:[ ^self error: 'The component named by ', (anIdentifier printString), ' does not exist.' ].
		comp activeFaultMode: aMode.].!

component: anIdentifier
	"Returns the component named by anIdentifier"

	| comp |
	ActualSystem isNil
	ifTrue:[ self error: 'An actual system must exist before searching for a component.' ]
	ifFalse:[ 
		comp := ActualSystem componentNamed: anIdentifier.
		comp isNil ifTrue:[ ^self error: 'The component named by ', (anIdentifier printString), ' does not exist.' ].
		^comp ].!

component: anIdentifier faultProbability: aValue for: aMode
	"Sets the fault probability of the given componenten mode to the given value."

	| comp |
	ActualSystem isNil
	ifTrue:[ ^self error: 'An actual system must exist before searching for a component.' ]
	ifFalse:[ 
		comp := ActualSystem componentNamed: anIdentifier.
		comp isNil ifTrue:[ ^self error: 'The component named by ', (anIdentifier printString), ' does not exist.' ].
		comp faultProbability: aValue for: aMode.].!

component: anIdentifier loadSD: aFilename using: anArray
	"Adds an hierarchic component to the actual system. The system
	description is loaded from the given file. This method should be 
	overwritten by my subclasses."!

component: anIdentifier repair: anObject for: aMode
	"Sets the repair information of the given componenten mode to the given value."

	| comp |
	ActualSystem isNil
	ifTrue:[ ^self error: 'An actual system must exist before searching for a component.' ]
	ifFalse:[ 
		comp := ActualSystem componentNamed: anIdentifier.
		comp isNil ifTrue:[ ^self error: 'The component named by ', (anIdentifier printString), ' does not exist.' ].
		comp repair: anObject for: aMode.].!

component: anIdentifier repairCost: aValue for: aMode
	"Sets the repair information of the given componenten mode to the given value."

	| comp |
	ActualSystem isNil
	ifTrue:[ ^self error: 'An actual system must exist before searching for a component.' ]
	ifFalse:[ 
		comp := ActualSystem componentNamed: anIdentifier.
		comp isNil ifTrue:[ ^self error: 'The component named by ', (anIdentifier printString), ' does not exist.' ].
		comp repairCost: aValue for: aMode.].!

component: anIdentifier type: aCompClass
	"Adds a component to the actual system. This method should be overwritten
	by my subclasses."!

component: anIdentifier type: aCompClass argument: anArgument
	"Adds a component to the actual system. anArgument can be additionally specified."

	self subclassResponsibility!

connect: aConnId withPort: aPortId at: aCompId
	"Connects a connection with a port of a component. This method should be overwritten
	by my subclasses."!

connection: anIdentifier
	"Adds a connection to the actual system. This method should be overwritten
	by my subclasses."!

connection: aConnectionId observationCost: aValue
	"Sets the cost for observing the given connection to the given value."

	| conn |
	ActualSystem isNil
	ifTrue:[ ^self error: 'An actual system must exist before using it.' ]
	ifFalse:[
		conn := self actualSystem connectionNamed: aConnectionId.
		conn isNil
		ifTrue:[ ^self error: 'The connection ', (aConnectionId printString), ' is not defined.' ]
		ifFalse:[ conn observationCost: aValue ].].!

connection: anIdentifier type: aVariableType
	"Adds a connection to the actual system. This method should be overwritten
	by my subclasses."!

diagnosis: anArray
	"Adds the diagnosis stored in anArray to the current diagnoses. "

	| diag |
	ActualSystem isNil
	ifTrue:[ self error: 'An actual system must exist before assigning diagnoses to it.' ]
	ifFalse:[
		diag := ExtendedDiagnosis new.
		anArray do:[ :cmaArray |
			diag
				addComponent: (ActualSystem componentNamed: (cmaArray at: 2))
				mode: (cmaArray at: 1).].
		ActualSystem diagnoses add: diag.
		^ActualSystem diagnoses ]!

inputConnection: anIdentifier
	"Adds an input connection to the actual system. This method should be overwritten
	by my subclasses."!

inputConnection: anIdentifier type: aVariableType
	"Adds an input connection to the actual system. This method should be overwritten
	by my subclasses."!

isNotObservable: aConnectionId
	"Sets the given connection to the not observeable state. Only observeable connections are
	shown as result of the measurement selection procedure."

	| conn |
	ActualSystem isNil
	ifTrue:[ ^self error: 'An actual system must exist before using it.' ]
	ifFalse:[
		conn := self actualSystem connectionNamed: aConnectionId.
		conn isNil
		ifTrue:[ ^self error: 'The connection ', (aConnectionId printString), ' is not defined.' ]
		ifFalse:[ conn setToNotObservable ].].!

isObservable: aConnectionId
	"Sets the given connection to the observeable state. Only observeable connections are
	shown as result of the measurement selection procedure."

	| conn |
	ActualSystem isNil
	ifTrue:[ ^self error: 'An actual system must exist before using it.' ]
	ifFalse:[
		conn := self actualSystem connectionNamed: aConnectionId.
		conn isNil
		ifTrue:[ ^self error: 'The connection ', (aConnectionId printString), ' is not defined.' ]
		ifFalse:[ conn setToObservable ].].!

maxDiagnosisSize: aValue
	"Stores the given parameter in the actual diagnosis system."

	ActualSystem isNil
	ifTrue:[ self error: 'An actual system must exist before assigning parameters to it.' ]
	ifFalse:[ ^ActualSystem maxDiagnosisSize: aValue ]!

maxFMEASize: aValue
	"Stores the given parameter in the actual diagnosis system."

	ActualSystem isNil
	ifTrue:[ self error: 'An actual system must exist before assigning parameters to it.' ]
	ifFalse:[ ^ActualSystem maxFMEASize: aValue ]!

maxNumberOfDiagnoses: aValue
	"Stores the given parameter in the actual diagnosis system."

	ActualSystem isNil
	ifTrue:[ self error: 'An actual system must exist before assigning parameters to it.' ]
	ifFalse:[ ^ActualSystem maxNumberOfDiagnoses: aValue ]!

observe: aValue on: aConnectionIdentifier
	"Stores the given observation in the actual diagnosis system."

	ActualSystem isNil
	ifTrue:[ self error: 'An actual system must exist before assigning an observation to it.' ]
	ifFalse:[ ^ActualSystem observation: aValue on: (ActualSystem connectionNamed: aConnectionIdentifier) ]!

outputConnection: anIdentifier
	"Adds an output connection to the actual system. This method should be overwritten
	by my subclasses."!

outputConnection: anIdentifier type: aVariableType
	"Adds an output connection to the actual system. This method should be overwritten
	by my subclasses."!

passiveFaultMode: aMode for: anIdentifier
	"Sets the given mode of the given component named by anIdentifier to the
	passive state.."

	| comp |
	ActualSystem isNil
	ifTrue:[ ^self error: 'An actual system must exist before searching for a component.' ]
	ifFalse:[ 
		comp := ActualSystem componentNamed: anIdentifier.
		comp isNil ifTrue:[ ^self error: 'The component named by ', (anIdentifier printString), ' does not exist.' ].
		comp passiveFaultMode: aMode.].!

systemName: anIdentifier

	ActualSystem isNil
	ifTrue:[ self error: 'An actual system must exist before assigning information to it.' ]
	ifFalse:[ ^ActualSystem identifier: anIdentifier ].! !

!DiagnosisSystem class methodsFor: 'class-initialization'!

defaultComponenClass

	^DiagnosisComponent!

defaultConnectionClass

	^DiagnosisConnection!

defaultTheoremProverClass
	"This method should be overwritten by my subclasses."

	^nil! !

DiagnosisSystem subclass: #DiagnosisSystemHSDag
	instanceVariableNames: 'hsdag '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Systems'!


!DiagnosisSystemHSDag methodsFor: 'initialize-release'!

initialize

	super initialize.
	hsdag := nil.
	^self! !

!DiagnosisSystemHSDag methodsFor: 'diagnosis'!

getConflictSet: anAssumptionSet
	"Answers a (minimal) choice set. This message is used by the hitting
	set generator for performing the diagnosis task on self.
	Because of the introduction of fault models a choice must
	be answered instead of a conflict."

	self subclassResponsibility!

isAssumeable: anAssumptionSet

	self subclassResponsibility!

makeDiagnosis
	"Starts the diagnosis procedure."

	actualDiagnosis := true.
	self setKnownComponentModes.
	hsdag := HSDagMultipleFM new
				createHSDag: self
				with: maxDiagnosisSize
				with: maxNumberOfDiagnoses.
	diagnoses := hsdag answerDiagnoses.
	^diagnoses!

makeDiagnosisWithFaultProbabilities
	"Starts the diagnosis procedure. Diagnoses are ordered by their probability
	values."

	actualDiagnosis := true.
	self setKnownComponentModes.
	hsdag := HSDagMultipleFM
			createHSDagUsingFaultProbabilities: self
			withNumberOfDiagnoses: maxNumberOfDiagnoses.
	diagnoses := hsdag answerDiagnosesProbability asSortedCollection reverse.
	^diagnoses!

makeIterativeDiagnosis
	"Starts the diagnosis procedure."

	actualDiagnosis := true.
	self setKnownComponentModes.
	hsdag := HSDagMultipleFM new
				iterativeCreateHSDag: self
				with: maxDiagnosisSize
				with: maxNumberOfDiagnoses.
	diagnoses := hsdag answerDiagnoses.
	^diagnoses!

measurementSelection
	"Starts the measurement selection procedure. Answers a dictionary containing connections
	and associated value indicating importance. A higher value means higher importance."

	| msDictionary result size entropy |
	msDictionary := Dictionary new.
"Compute value occurences ..."
	diagnoses do:[ :diag |
		(self simulate: diag) keysAndValuesDo:[ :conn :value | | vc vcValue |
			(conn storesMultipleValues)
			ifTrue:[
				value do:[ :val |
					vc := msDictionary at: conn ifAbsent:[ msDictionary at: conn put: (Dictionary new) ].
					vcValue := vc at: val ifAbsent:[ 0.0d ].
					diag storesProbabilities
					ifTrue:[ vc at: val put: (vcValue + (diag probability)).]
					ifFalse:[ vc at: val put: (vcValue + 1).].].]
			ifFalse:[
				(value isNil)
				ifFalse:[
					vc := msDictionary at: conn ifAbsent:[ msDictionary at: conn put: (Dictionary new) ].
					vcValue := vc at: value ifAbsent:[ 0.0d ].
					diag storesProbabilities
					ifTrue:[ vc at: value put: (vcValue + (diag probability)).]
					ifFalse:[ vc at: value put: (vcValue + 1).].].].].].
"Examine occurences ..."
	result := SortedCollection sortBlock:[ :x :y | x value > (y value) ].
	msDictionary keysAndValuesDo:[ :conn :valueDictionary |
		size := 0.
		valueDictionary do:[ :num |
			size := size + num. ].
		entropy := 0.0d.
		valueDictionary keysAndValuesDo:[ :value :num | | valD |
			valD := (num / size) asDouble.
			entropy := entropy + (( valD * (valD ln)) abs asDouble).].
		(conn isObservable)
		ifTrue:[ result add: conn -> entropy. ].].
	^result!

proveConsistency: anAssumptionSet
	"Answers true if the given assumptions do not cause a contradiction and false otherwise.
	Note that the given assumption set do not contain components which are assumed
	to behave correct."

	self subclassResponsibility!

setKnownComponentModes
	"Sets the modes of components not included in the focus to the
	default nab mode."

	(components - focus) do:[ :comp | comp setMode: (comp nabMode) ].
	^self! !

DiagnosisSystemHSDag subclass: #CPDiagnosisSystem
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Systems'!


!CPDiagnosisSystem methodsFor: 'diagnosis'!

getConflictSet: anAssumptionSet
	"Answers a (minimal) choice set. This message is used by the hitting
	set generator for performing the diagnosis task on self.
	Because of the introduction of fault models a choice must
	be answered instead of a conflict."

	|  result conflict newSet |
	result := self proveConsistency: anAssumptionSet.
	(result isEndLabel)
	ifTrue:[^result]
	ifFalse:[
		conflict := ExtendedDiagnosis new.
		newSet := result copy.
		result do:[ :el |
			(self isAssumeable: (newSet remove: el;yourself))
			ifTrue:[
				newSet add:el.
				conflict add: el. ].].
		^conflict asExtendedChoice.].!

isAssumeable: anAssumptionSet

	self reset.
	anAssumptionSet setComponentMode.
	self loadTheoremProverWithObservations.
	^self isConsistent!

loadTheoremProverWithInputObservations
	"Loads the observations related to inputs into the theorem prover and starts propagation."

	| inputs |
	inputs := self allInputConnections.
	observations keysAndValuesDo:[ :con :val |
		(inputs includes: con)
		ifTrue:[ con setObservation: val. ].].
	theoremProver propagation.!

loadTheoremProverWithObservations
	"Loads the observations into the theorem prover and starts propagation."

	observations keysAndValuesDo:[ :con :val | con setObservation: val. ].
	theoremProver propagation.!

proveConsistency: anAssumptionSet
	"Answers true if the given assumptions do not cause a contradiction and false otherwise.
	Note that the given assumption set do not contain components which are assumed
	to behave correct."

	| comps |
	self reset.
"Set component modes ..."
	comps := ExtendedDiagnosis withOKComponents: (( self focus asSet) - (anAssumptionSet onlyComponents)).
	comps setComponentMode.
	anAssumptionSet setComponentMode.
"Do theorem proving ..."
	self loadTheoremProverWithObservations.
"Check consistency ..."
	self isConsistent
	ifFalse:[ comps addAll: anAssumptionSet. ^comps ]
	ifTrue:[ ^HSDagNodeMultipleFM endLabel ].!

reset
	"Sets the theorem prover to the reset state."

	components do:[ :comp |
		(focus includes: comp)
		ifTrue:[ comp identifier reset.]
		ifFalse:[ comp identifier softReset ].].
	^theoremProver softReset! !

!CPDiagnosisSystem methodsFor: 'simulation'!

computeOutputValues: aDiagnosis
	"Computes the values of all output connections assuming that the components
	behave like given by aDiagnosis."

	| result outputs |
	outputs := self allOutputConnections.
	self reset.
	components do:[ :comp | comp setMode: (comp nabMode) ].
	aDiagnosis setComponentMode.
	self loadTheoremProverWithInputObservations.
	result := Dictionary new.
	connections do:[ :conn |
		(outputs includes: conn)
		ifTrue:[
			(conn storesMultipleValues)
			ifTrue:[
				result at: conn put: (Set new).
				conn allValues do:[ :val | (result at: conn) add: val ].]
			ifFalse:[ result at: conn put: (conn value).].].].
	^result!

simulate: aDiagnosis
	"Computes the values of all connections assuming that the components
	behave like given by aDiagnosis."

	| result |
	self reset.
	components do:[ :comp | comp setMode: (comp nabMode) ].
	aDiagnosis setComponentMode.
	self loadTheoremProverWithObservations.
	result := Dictionary new.
	connections do:[ :conn |
		(conn storesMultipleValues)
		ifTrue:[
			result at: conn put: (Set new).
			conn allValues do:[ :val | (result at: conn) add: val ].]
		ifFalse:[ result at: conn put: (conn value).].].
	^result! !

!CPDiagnosisSystem methodsFor: 'accessing'!

addComponent: aDiagnosisComponent

	compNames at: (aDiagnosisComponent named) put: aDiagnosisComponent.
	^components add: aDiagnosisComponent!

addConnection: aDiagnosisConnection

	connNames at: (aDiagnosisConnection named) put: aDiagnosisConnection.
	^connections add: aDiagnosisConnection! !

!CPDiagnosisSystem methodsFor: 'testcase-computation'!

computeOBSForOutputs: anOutputCollection
	"Computes a collection of observations containing an output vector for
	the given observations previously stored in self. The output vector is computed randomly by
	assuming that at least one value is wrong. A dictionary associating a value to
	every element of anOutputCollection is returned as result."

	| results conns value label rand |
	results := Dictionary new.
	conns := self simulate.
	label := true.
	rand := Random new.
	rand next; next; next.
	anOutputCollection do:[ :conn |
		value := conns at: conn ifAbsent:[ nil ].	
		value isNil
		ifFalse:[
			((rand next < 0.5) and:[ label ])
			ifTrue:[
				label := false.
				conn storesMultipleValues
				ifTrue:[
					results at: conn put: (Set new).
					value do:[ :val | (results at: conn) add: (val not). ].]
				ifFalse:[ results at: conn put: (value not).].]
			ifFalse:[ results at: conn put: value.].].].
	label ifTrue:[
		anOutputCollection do:[ :conn |
			value := conns at: conn ifAbsent:[ nil ].	
			value isNil
			ifFalse:[
				label
				ifTrue:[
					label := false.
					conn storesMultipleValues
					ifTrue:[
						results at: conn put: (Set new).
						value do:[ :val | (results at: conn) add: (val not). ].]
					ifFalse:[ results at: conn put: (value not).].].].].].
	^results! !

!CPDiagnosisSystem methodsFor: 'public-accessing-structure'!

connectionFor: aCompId withPort: aPortId
	"Returns the identifier of the connection connected with the given
	component using the given port."

	| comp |
	comp := self componentNamed: aCompId.
	comp isNil
	ifFalse:[ ^comp connectionFor: aPortId ].!

disconnect: aCompId withPort: aPortId
	"Removes the connection from the given port of the given
	component."

	| comp |
	comp := self componentNamed: aCompId.
	comp isNil ifFalse:[ ^comp disconnectPort: aPortId ].! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CPDiagnosisSystem class
	instanceVariableNames: ''!


!CPDiagnosisSystem class methodsFor: 'class-initialization'!

defaultComponenClass

	^CPDiagnosisComponent!

defaultConnectionClass

	^CPDiagnosisConnection!

defaultTheoremProverClass
	"Answers the default theorem prover class."

	^ConstraintPropagator! !

!CPDiagnosisSystem class methodsFor: 'file-accessing-private'!

component: anIdentifier loadSD: aFilename using: anArray
	"Adds an hierarchic component to the actual system. The system
	description is loaded from the given file. anArray contains the relations
	between variables used in aFilename and ports."

	| oldSystem comp dc dict |
	oldSystem := DiagnosisSystem actualSystem.
	(aFilename asFilename exists)
	ifTrue:[
		DiagnosisSystem actualSystem: (self new).
		DiagnosisSystem actualSystem theoremProver: (self defaultTheoremProverClass new).
		aFilename asFilename fileIn.
		comp := SCCHierarchic identifier: anIdentifier.
		comp cp: (DiagnosisSystem actualSystem theoremProver).
		oldSystem theoremProver addConstraint: comp.
		dict := Dictionary new.
		anArray do:[ :rel | dict at: (rel at: 1) put: ((DiagnosisSystem actualSystem connectionNamed: (rel at: 2)) identifier). ].
		comp associations: dict.
		comp createPorts: (comp associations keys).
		dc := comp asDiagnosisComponent.
		oldSystem addComponent: dc.
		oldSystem addToFocus: dc.
		dc diagnosisSystem: oldSystem.
		^DiagnosisSystem actualSystem: oldSystem ]
	ifFalse:[ self error: 'The file ', (aFilename printString), ' does not exist or is not readable' ].!

component: anIdentifier type: aCompClass
	"Adds a component to the actual system."

	| comp dc |
	comp := aCompClass identifier: anIdentifier.
	ActualSystem theoremProver addConstraint: comp.
	dc := comp asDiagnosisComponent.
	ActualSystem addComponent: dc.
	ActualSystem addToFocus: dc.
	dc diagnosisSystem: ActualSystem.!

component: anIdentifier type: aCompClass argument: anArgument
	"Adds a component to the actual system."

	| comp dc |
	comp := aCompClass identifier: anIdentifier with: anArgument.
	ActualSystem theoremProver addConstraint: comp.
	dc := comp asDiagnosisComponent.
	ActualSystem addComponent: dc.
	ActualSystem addToFocus: dc.
	dc diagnosisSystem: ActualSystem.!

connect: aConnId withPort: aPortId at: aCompId
	"Connects a connection with a port of a component."

	| comp conn |
	conn := (ActualSystem connectionNamed: aConnId) identifier.
	comp := (ActualSystem componentNamed: aCompId) identifier.
	comp connect: conn usingPort: aPortId.!

connection: anIdentifier
	"Adds a connection to the actual system."

	^self connection: anIdentifier type: SConstraintVariable!

connection: anIdentifier type: aVariableType
	"Adds a connection to the actual system."

	| con dc |
	con := aVariableType identifier: anIdentifier.
	ActualSystem theoremProver addVariable: con.
	dc := con asDiagnosisConnection.
	ActualSystem addConnection: dc.
	dc diagnosisSystem: ActualSystem.!

inputConnection: anIdentifier
	"Adds an input connection to the actual system."

	^self inputConnection: anIdentifier type: SConstraintVariable!

inputConnection: anIdentifier type: aVariableType
	"Adds an input connection to the actual system."

	| con dc |
	con := aVariableType identifier: anIdentifier.
	con asInput.
	ActualSystem theoremProver addVariable: con.
	dc := con asDiagnosisConnection.
	ActualSystem addConnection: dc.
	dc diagnosisSystem: ActualSystem.!

outputConnection: anIdentifier
	"Adds an output connection to the actual system."

	^self outputConnection: anIdentifier type: SConstraintVariable!

outputConnection: anIdentifier type: aVariableType
	"Adds an output connection to the actual system."

	| con dc |
	con := aVariableType identifier: anIdentifier.
	con asOutput.
	ActualSystem theoremProver addVariable: con.
	dc := con asDiagnosisConnection.
	ActualSystem addConnection: dc.
	dc diagnosisSystem: ActualSystem.! !

DiagnosisObject subclass: #DiagnosisComponent
	instanceVariableNames: 'activeModes modeProbabilities diagnosisSystem repair repairCost '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Systems'!


!DiagnosisComponent methodsFor: 'printing'!

name

	^identifier printString!

printOn: aStream
	"Writes self on aStream."

	identifier printOn: aStream.!

shortName

	^identifier printString! !

!DiagnosisComponent methodsFor: 'private'!

checkModeProbabilities
	"Answers true if the sum of all mode probabilites is not greater then 1 and
	false otherwise."

	| sum |
	sum := 0.0d.
	modeProbabilities associationsDo:[ :assoc |
		sum := sum + (assoc value). ].
	^sum <= 1.0d!

setNABModeProbability
	"Computes the not abnormal mode probability out of the other probabilities."

	| sum |
	modeProbabilities removeKey: (self nabMode).
	sum := 0.0d.
	modeProbabilities associationsDo:[ :assoc |
		sum := sum + (assoc value). ].
	modeProbabilities at: (self nabMode) put: (1.0d - sum).
	^sum! !

!DiagnosisComponent methodsFor: 'public-accessing'!

activeFaultMode: aMode
	"Sets the given mode to active."

	^activeModes add: aMode!

canBeEntered
	"Answers true if self can be used for hierarchic diagnosis."

	^self subclassResponsibility!

enter
	"Answers a new instance of diagnosis system or a subclass."

	^self subclassResponsibility!

faultModes
	"Answers a collection of implemented fault modes."

	^self subclassResponsibility!

faultProbability
	"Get the probability of the abnormal fault mode."

	^self faultProbabilityValue.!

faultProbability: aValue
	"Sets the probability of the abnormal fault mode to the given value."

	^self faultProbability: aValue for: #ab!

faultProbability: aValue for: aMode
	"Sets the probability of the given fault mode to the given value."

	| prob |
	((aValue < 0) or:[ aValue > 1]) ifTrue:[ self error: 'A probability value must be between 0 and 1.' ].
	prob := modeProbabilities at: aMode ifAbsent:[ 0.0d ].
	modeProbabilities at: aMode put: (aValue asDouble).
	self setNABModeProbability.
	(self checkModeProbabilities)
	ifFalse:[ 
		modeProbabilities at: aMode put: prob.
		self setNABModeProbability.
		self error: 'The sum of all mode probabilities must not exceed 1'. ].!

faultProbabilityFor: aMode
	"Get the probability of the given fault mode."

	^self faultProbabilityValueFor: aMode!

isActiveFaultMode: aMode
	"Test, if the given mode is active."

	^activeModes includes: aMode!

isPassiveFaultMode: aMode
	"Test, if the given mode is passive."

	^(activeModes includes: aMode) not!

passiveFaultMode: aMode
	"Sets the given mode to passive."

	^activeModes remove: aMode ifAbsent:[ nil ]! !

!DiagnosisComponent methodsFor: 'initialize-release'!

initialize

	identifier := nil.
	activeModes := Set new.
	modeProbabilities := Dictionary new.
	repair := Dictionary new.
	repairCost := Dictionary new.
	^self! !

!DiagnosisComponent methodsFor: 'accessing'!

activeModes

	^activeModes!

diagnosisSystem

	^diagnosisSystem!

diagnosisSystem: aSystem

	^diagnosisSystem := aSystem!

faultProbabilityValue
	"Answers the fault probability of the #ab mode."

	^self faultProbabilityValueFor: #ab!

faultProbabilityValueFor: aMode
	"Answers the fault probability of the given mode."

	^modeProbabilities at: aMode ifAbsent:[ 0.0d ].!

modes

	^activeModes!

named

	^identifier printString!

repair

	^self repairFor: #ab!

repair: anObject

	^self repair: anObject for: #ab!

repair: anObject for: aMode

	^repair at: aMode put: anObject!

repairCost

	^self repairCostFor: #ab!

repairCost: aValue

	^self repairCost: aValue for: #ab!

repairCost: aValue for: aMode

	^repairCost at: aMode put: aValue!

repairCostFor: aMode

	^repairCost at: aMode ifAbsent:[ 0.0d ]!

repairFor: aMode

	^repair at: aMode ifAbsent:[ nil ]! !

!DiagnosisComponent methodsFor: 'diagnosis'!

nabMode

	^#nab!

resetMode
	"This method resets the component mode to a default state."

	^self!

resetToDefaults!

setMode: aMode
	"Sets self to the given mode."

	^self subclassResponsibility! !

!DiagnosisComponent methodsFor: 'testing'!

isConnected
	"Answers true if self is connected."

	self subclassResponsibility! !

DiagnosisComponent subclass: #CPDiagnosisComponent
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Systems'!


!CPDiagnosisComponent methodsFor: 'public-accessing'!

canBeEntered
	"Answers true if self can be used for hierarchic diagnosis."

	^identifier canBeEntered!

connectionFor: aPortId
	"Answers the connection connected with the given port."

	^identifier connectedVariable: aPortId!

disconnectPort: aPortId
	"Removes the connection used at the given port."

	^identifier disconnectPort: aPortId!

enter
	"Answers a new instance of diagnosis system or a subclass."

	self canBeEntered
	ifTrue:[ ^identifier enter asDiagnosisSystem ]
	ifFalse:[ ^nil ].!

faultModes
	"Answers a collection of implemented fault modes."

	^identifier modes asSet! !

!CPDiagnosisComponent methodsFor: 'diagnosis'!

nabMode

	^identifier defaultNABMode!

resetToDefaults

	| fm |
	fm := identifier modes.
	activeModes := Set new.
	modeProbabilities := Dictionary new.
	fm do:[ :mode |
		activeModes add: mode.
		modeProbabilities at: mode put: (identifier faultProbabilityValueFor: mode).
		repair at: mode put: (identifier repairFor: mode).
		repairCost at: mode put: (identifier repairCostFor: mode).].
	^self!

setMode: aMode
	"Sets the component to the given mode."

	^identifier mode: aMode! !

!CPDiagnosisComponent methodsFor: 'accessing'!

identifier: anObject

	identifier := anObject.
	^self resetToDefaults!

modes

	^activeModes!

named

	^identifier identifier!

value: aValue
	"This method should only be used if self contains an instance of SCCConstant."

	(identifier class = SCCConstant)
	ifTrue:[ ^identifier value: aValue ].! !

!CPDiagnosisComponent methodsFor: 'printing'!

name

	^identifier identifier printString! !

!CPDiagnosisComponent methodsFor: 'testing'!

isConnected
	"Answers true if self is connected."

	^identifier isConnected! !
DiagnosisObject initialize!

GenericDiagnosisSystem initialize!


