Set variableSubclass: #Diagnose
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Kernel'!
Diagnose comment:
'(c) 1995 F. Wotawa TU Wien, Institut fuer Informationssysteme

This class implements all relevant methods for diagnosis. This includes a conversion to diagnosis
with probabilities and comparing methods. Instances are used in instances of DiagnoseSet and
while HSDag generation.'!


!Diagnose methodsFor: 'printing'!

asPrinterString
	"Answers self using the printer specific format"

	^self printString!

hardcopy
	"Prints the diagnosis list to the standard printer"

	| str |
	str := 'Diagnosis: ', (self asPrinterString).
	(str asComposedText) hardcopy.!

printOn: aStream

	| ch |

	ch := ''.
	(self isEmpty)
	ifTrue:[
		aStream nextPutAll: 'Empty Diagnose'.]
	ifFalse:[
		self do:[ :comp |
			aStream nextPutAll:ch; nextPutAll: (comp shortName).
			ch := ', '.].].! !

!Diagnose methodsFor: 'set-operations'!

equal: aDiagnosis

	(self size = (aDiagnosis size))
	ifTrue:[
		self do:[ :el | (aDiagnosis includes: el) ifFalse:[ ^false ].].
		^true ]
	ifFalse:[ ^false ].!

intersect: aCollection
	"Computes a new instance of self containing intersecting elements."

	| result |
	result := self species new.
	aCollection do:[ :el |
		(self includes: el) ifTrue:[ result add: el ].].
	^result!

union: aCollection
	"Answers a new instance containing the union."

	| result |
	result := self species new.
	result addAll: self.
	result addAll: aCollection.
	^result! !

!Diagnose methodsFor: 'testing'!

< aDiagnose

	^self size < (aDiagnose size)!

<= aDiagnose

	^self size <= (aDiagnose size)!

> aDiagnose

	^self size > (aDiagnose size)!

>= aDiagnose

	^self size >= (aDiagnose size)!

isEmptyDiagnosis

	^self isEmpty!

subset: aCollection
	"Answers true if self is a subset of aCollection and false otherwise."

	(self size < (aCollection size))
	ifTrue:[
		self do:[ :el | (aCollection includes: el) ifFalse:[ ^false ].].
		^true].
	^false! !

!Diagnose methodsFor: 'converting'!

asSortedCollection
	"Converts self to an sorted collection. Diagnosis components are
	ordered using their fault probabilities."

	| result |
	result := SortedCollection new.
	result sortBlock: [:x :y | (x faultProbability) > (y faultProbability)].
	result addAll: self.
	^result!

convertToDiagnoseProbability: aValue
	"Converts self to an instance of DiagnoseProbability using aValue.
	aValue must be the probability of self."

	| result |
	result := DiagnoseProbability new: (self size).
	result addAll: self.
	result probability: aValue.

	^result! !

!Diagnose methodsFor: 'comparing'!

= aDiagnose

	^self equal: aDiagnose!

hash

	| hv |
	hv := 17717.
	self do:[ :el | hv := hv bitXor: (el hash) ].
	^hv! !

!Diagnose methodsFor: 'accessing'!

first

	self do:[ :el | ^el ].
	^nil! !

Diagnose variableSubclass: #DiagnoseProbability
	instanceVariableNames: 'probability '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Kernel'!
DiagnoseProbability comment:
'(c) 1995 F. Wotawa TU Wien, Institut fuer Informationssysteme

This class implements diagnoses with given probability. '!


!DiagnoseProbability methodsFor: 'accessing'!

probability

	^probability!

probability: aValue

	^probability := aValue! !

!DiagnoseProbability methodsFor: 'initialization'!

init

	self probability: (1.0).

	^self! !

!DiagnoseProbability methodsFor: 'printing'!

printOn: aStream

	| ch |

	ch := ''.
	(self isEmpty)
	ifTrue:[
		aStream nextPutAll: 'Empty Diagnose'.]
	ifFalse:[
		self do:[ :comp |
			aStream nextPutAll:ch; nextPutAll: (comp shortName).
			ch := ', '.].
		aStream nextPutAll: (' [ ', (self probability printString), ' ] ').].
	aStream cr.! !

!DiagnoseProbability methodsFor: 'testing'!

< aDiagnose

	^self probability < (aDiagnose probability)!

<= aDiagnose

	^self probability <= (aDiagnose probability)!

> aDiagnose

	^self probability > (aDiagnose probability)!

>= aDiagnose

	^self probability >= (aDiagnose probability)! !

!DiagnoseProbability methodsFor: 'fileIn/Out'!

representBinaryOn: binWriter
	"This method have to be changed because a new instance
	variable has been introduced. Values of the instance variable
	probability must be stored, too."

	| creator args |

	args := Array new: 2.
	args at:1 put: ((self isEmpty ifTrue: [#()] ifFalse: [self asArray])).
	args at:2 put: (self probability).

	creator := MessageSend
			receiver: self class
			selector: #withAll:and:
			arguments: args.
	^binWriter expectCycles
		ifTrue: [BinaryObjectStorage indexImportSendSelf ->
			(Array
				with: #createSpecialObject:
				with: (BOSSSpecialObjectLoader message: creator))]
		ifFalse: [creator]! !

!DiagnoseProbability methodsFor: 'conversion'!

convertToDiagnose
	"Reconverts self to an instance of Diagnose.
	The probability value is removed."

	| result |
	result := Diagnose new: (self size).
	self do:[ :comp |
		result add: comp.].
	^result! !

!DiagnoseProbability methodsFor: 'converting'!

convertToDiagnoseProbability: aValue
	
	^self! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

DiagnoseProbability class
	instanceVariableNames: ''!


!DiagnoseProbability class methodsFor: 'instance creation'!

new

	^super new init!

new: aValue

	^(super new: aValue) init!

withAll: anArray and: prob

	| result |
	result := super new.
	result addAll: anArray.
	result probability: prob.
	^result! !

Set variableSubclass: #DiagnoseSet
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Kernel'!
DiagnoseSet comment:
'(c) 1995 F. Wotawa TU Wien, Institut fuer Informationssysteme

This class implements a set containing only diagnoses. Usefull methods for diagnosis sets are implemented
Instances of this class are used while generating a HSDag. A complete package for diagnosis should
use this class.'!


!DiagnoseSet methodsFor: 'file I/O'!

load: aString
	"Answers an instance of self that has been stored
	in a file with name aString"

	^self class load: aString!

loadFromFile
	"Loades a BOSS file and insert the contents to the diagnosis list."

	^self class loadFromFile!

save: aString
	"Save self in a file with name aString"

	| stream bos |
	stream := aString asFilename writeStream.
	bos := BinaryObjectStorage onNew: stream.
	[bos nextPut: self]
		valueNowOrOnUnwindDo: [ bos close ].!

saveToFile
	"Saves the diagnosis list as BOSS file."

	| file |
	file := Dialog 
		requestFileName: 'Store in .. '
		default: 'diag.DD'
		version: #new.
	(file isNil) | (file isEmpty)
	ifFalse:[
		self save: file.
		Dialog warn: 'Diagnosis list saved'. ].! !

!DiagnoseSet methodsFor: 'copying'!

copyEmpty

	^self copyEmpty: (self size)! !

!DiagnoseSet methodsFor: 'converting'!

asSortedCollection
	"Answers a sorted collection"

	| result |
	result := SortedCollection new.
	result sortBlock:[ :x :y | x > y].
	result addAll: self.
	^result!

onlyComponents
	"Answers a diagnosis set containing diagnosis using only components."

	| result |
	result := self species new.
	self do:[ :diag | | newDiag |
		newDiag := diag onlyComponents.
		(result includes: newDiag) ifFalse:[ result add: (diag onlyComponents)].].
	^result!

withProbabilities

	| ds |
	ds := self species new.
	self do:[ :diag | ds add: (diag convertToDiagnoseProbability) ].
	^ds! !

!DiagnoseSet methodsFor: 'testing'!

containsEmptyDiagnosis

	self do:[ :el |
		el isEmptyDiagnosis ifTrue:[ ^true ].].
	^false!

existsSubset: aDiagnose
	"Answers true if self includes at least one diagnoses that is a 
	subset of aDiagnose. Returns false otherwise."

	self do:[ :diag |
		(diag subset: aDiagnose) ifTrue:[^true].].
	^false!

existsSuperset: aDiagnose
	"Answers true if self includes at least one diagnoses that is a 
	superset of aDiagnose. Returns false otherwise."

	self do:[ :diag |
		(aDiagnose subset: diag) ifTrue:[^true].].
	^false!

includes: aDiagnose
	"Answer whether aDiagnose is one of the receiver's elements."

	self do: [:each | (aDiagnose equal: each) ifTrue: [^true]].
	^false! !

!DiagnoseSet methodsFor: 'set operations'!

combineWith: aDiagSet
	"Builds a new diagnosis set out of all elements of self and the
	elements of aDiagSet."

	| result |
	result := self intersect: aDiagSet.
	self do:[ :diag |
		(aDiagSet existsSuperset: diag)
		ifTrue:[ result add: diag ].].
	aDiagSet do:[ :diag |
		(self existsSuperset: diag)
		ifTrue:[ result add: diag ].].
	^result!

minimize
	"Answers a new diagnoses set containing of diagnoses with the
	property that there is no diagnosis that has a subset in self."

	| ds |
	ds := self class new.
	self do:[ :diag |
		(self existsSubset: diag)
		ifFalse:[ ds add: diag ].].
	^ds!

reduceDiagnosisSize: aSize
	"Removes all diagnosis with a size greater than the given one."

	(self select:[ :comp | comp size > aSize ]) do:[ :rc | self remove: rc ].!

times: aDiagnoseSet

	| ds |
	ds := self species new.
	self isEmpty ifTrue:[ ^ds ].
	aDiagnoseSet isEmpty ifTrue:[ ^ds ].
	self do:[ :diag1 |
		aDiagnoseSet do:[ :diag2 |
			ds add: (diag1 union: diag2).].].
	^ds! !

!DiagnoseSet methodsFor: 'printing'!

hardcopy
	"Prints the diagnosis list to the standard printer"

	| str |
	self isEmpty
	ifTrue:[
		Dialog warn: 'Nothing here to print'.]
	ifFalse:[
		str := 'Diagnosis List:
---------------------------------------------------------------------
'.
		self do:[ :de |
			str := str , (de asPrinterString), '
'.].
		str := str ,'---------------------------------------------------------------------'.
		(str asComposedText) hardcopy.].! !

!DiagnoseSet methodsFor: 'accessing'!

first

	self do:[ :el | ^el ].
	^nil! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

DiagnoseSet class
	instanceVariableNames: ''!


!DiagnoseSet class methodsFor: 'file I/O'!

load: aString
	"Answers an instance of self that has been stored
	in a file with name aString"

	| file stream bos dl |
	file := aString asFilename.
	(file isReadable)
	ifTrue:[
		stream := file readStream.
		bos := BinaryObjectStorage onOldNoScan: stream.
		[ dl := bos contents ]
			valueNowOrOnUnwindDo:[ bos close ].].
	^dl at: 1!

loadFromFile
	"Loades a BOSS file and insert the contents to the diagnosis list."

	| file dl |
	file := Dialog 
		requestFileName: 'Load from .. '
		default: 'diag.DD'
		version: #old.
	(file isNil) | (file isEmpty)
	ifFalse:[
		dl := self load: file.
		^dl ].
	^nil! !

Object subclass: #HSDag
	instanceVariableNames: 'nodes endNodes topNode theoremProverCalls diagNumber '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Kernel'!
HSDag comment:
'(c) 1995 F. Wotawa TU Wien, Institut fuer Informationssysteme

This class implements the hitting set algorithm originaly introduced by Reiter, improved by
Greiner et al, and finaly improved by a master thesis from Roland Unger (with help from
Gerhard Friedrich). Node creation, reuse and dag pruning has been implemented.
The HSDag class uses several methods as an interface to a theorem prover. Classes
which implements a theorem prover must include one method

getConflictSet: aComponentSet		Answers the next conflict set using the
								component set. This method should return
								a minimal conflict set to decrease
								computation time.

If HSDag instances are used for finding diagnoses using fault probabilities the
theorem prover should also implement

diagComponents					Answers a set of diagnosis components
proveConsistency: aComponentSet 	Answers true if the component set is
								consistent with the rest of the system
								description.

Additionally every component used for diagnosis must include a method

faultProbability

to answer its fault probability.

Self uses the classes HSDagArc and HSDagNode as basic classes that implements 
the structure of an hitting set dag. An instance of HSDagNode is given as result
after applying the hs-algorithm.

nodes		<Contains all dag nodes>
endNodes		<Contains dag nodes that are labeled with an end label.>
topNode		<Contains the top node of the resulting dag.>
theoremProverCalls	<An integer counting the number of theorem prover calls
				while computing the HS tree>
diagNumber	<An integer containing the maximum number of diagnosis that
			should be computed>

HS dag generation should be invoked using the methods

(1) createHSDag: aTheoremProver with: diagSize with: numberOfDiagnoses
To compute the HS dag using a theorem prover the expected maximum
diagnosis size and the expected number of computed diagnosis.

(2) createHSDagUsingFaultProbabilities: aTheoremProver withNumberOfDiagnoses: aSize
To compute the HS dag using fault probabilities. The number of expected diagnosis must
be given by the calling object.

'!


!HSDag methodsFor: 'initialize-release'!

initialize

	topNode := nil.
	theoremProverCalls := 0.
	nodes := Set new.
	endNodes := Set new.
	^self! !

!HSDag methodsFor: 'accessing'!

endNodes

	^endNodes!

endNodes: aCollection

	^endNodes := aCollection!

nodes

	^nodes!

nodes: aCollection

	^nodes := aCollection!

topNode

	^topNode!

topNode: aNode

	^topNode := aNode! !

!HSDag methodsFor: 'dag generation'!

breadthFirst: aTheoremProver with: aSet with: aSF with: diagSize with: numberOfDiagnoses
	"Creates new nodes using aTheoremProver, a set of nodes and a ordered collection of
	previously created conflict sets. Pruning, node reusing and node closing is used to
	compute the smallest possible graph."

	| nSet |
	(diagSize <= 0) ifTrue:[ ^self topNode ].
	nSet := Set new.
	aSet do:[ :n |
		(self closeNode: n)
		ifFalse:[
			n label: nil.

			aSF do:[ :c |
				(c intersect: (n h)) isEmpty
				ifTrue:[ n label: c.].].

			(n label isNil)
			ifTrue:[
				theoremProverCalls := theoremProverCalls + 1.
				n label: (aTheoremProver getConflictSet: (n h)).
				(n label = (HSDagNode endLabel)) ifFalse:[
					aSF addLast: (n label).
					self pruneNode: n with: aSF.].].

			(n label = (HSDagNode noLabel))
			ifFalse:[
				(n label = (HSDagNode endLabel))
				ifTrue:[
					"Compute only numberOfDiagnoses end nodes."
					self endNodes add: n.
					(self endNodes size) >= numberOfDiagnoses
					ifTrue:[^self topNode.].]
				ifFalse:[
					(n label) do:[ :al | | nn |
						nn := self reuseNode: n with: al with: nSet.
						n next add: (HSDagArc node: nn label: al).].].].].].

	(nSet isEmpty) ifFalse:[^self breadthFirst: aTheoremProver with: nSet with: aSF with: (diagSize - 1) with: numberOfDiagnoses].

	^self topNode!

closeNode: aNode
	"A specific node is closed, e.g. no further calculation is done for this node
	if its history is a subset of a previously computed end node.
	This method answers true if aNode can be closed without loosing information
	and false otherwise."

	self endNodes do:[ :n |
		((n h) subset: (aNode h))
		ifTrue:[ aNode label: (HSDagNode closeLabel). ^true].].
	^false!

createHSDag: aTheoremProver with: diagSize with: numberOfDiagnoses
	"Creates a hitting set dag using a theorem prover loaded with SD, COMPS and OBS.
	The resulting hs dag size is smaller than diagSize."

	| top sf |
	top := HSDagNode newTop.
	self nodes add: top.
	self topNode: top.

	sf := OrderedCollection new.
	self breadthFirst: aTheoremProver with: (Set new add: top; yourself) with: sf with: (diagSize + 1) with: numberOfDiagnoses.

	^self!

createHSDagUsingFaultProbabilities: aTheoremProver withDiagSize: aSize withNumberOfDiagnoses: aNumber
	"Computes  end nodes which are related to diagnoses with highest probability. The number
	of computed end nodes is restricted by aNumber."

	| comps prob top |
	diagNumber := aNumber.
	comps := SortedCollection new.
	comps
		sortBlock: [ :x :y | x faultProbability <=  (y faultProbability).];
		addAll: (aTheoremProver diagComponents).
	prob := 1.0d.
	comps do:[ :comp |
		prob := prob * (1.0d - (comp faultProbability)).].
	top := HSDagNode newTop.
	self nodes add: top.
	self topNode: top.
	self topNode probability: prob.
	^self processUsingFaultProbabilities: top with: comps using: aTheoremProver!

createHSDagUsingFaultProbabilities: aTheoremProver withNumberOfDiagnoses: aNumber
	"Computes  end nodes which are related to diagnoses with highest probability. The number
	of computed end nodes is restricted by aNumber."

	^self createHSDagUsingFaultProbabilities: aTheoremProver withDiagSize: 1 withNumberOfDiagnoses: aNumber!

processUsingFaultProbabilities: aTopNode with: aSortedCollection using: aTheoremProver
	"This method implements the end node search using fault probabilities.
	No pruning or other optimization techniques are used."

	| newTopNode result |

	theoremProverCalls := theoremProverCalls + 1.

	(aTheoremProver proveConsistency: aSortedCollection) = (HSDagNode endLabel)
	ifTrue:[
		aTopNode label: (HSDagNode endLabel).
		diagNumber := diagNumber - 1.
		^self topNode.]
	ifFalse:[
		aTopNode label: aSortedCollection.
		aSortedCollection do:[ :comp |

			newTopNode := HSDagNode newNext: aTopNode.
			self nodes add: newTopNode.
			newTopNode h: ((aTopNode h copy) add: comp; yourself).
			newTopNode probability: (((aTopNode probability) * (comp faultProbability)) / (1.0d - (comp faultProbability))).

			aTopNode next add: (HSDagArc node: newTopNode label: comp).

			result := self processUsingFaultProbabilities: newTopNode 
				with: (aSortedCollection copy remove: comp; yourself) using: aTheoremProver.

			(diagNumber <= 0) ifTrue:[ ^result .].].].!

pruneNode: aNode with: aSF
	"A node can be pruned if it do not provide new information for
	the hitting set tree. "

	| ds |
	self nodes do:[ :n | | ol |
		(n label = (HSDagNode endLabel) | (n label = (HSDagNode closeLabel)) | (n label isNil))
		ifFalse:[
		((aNode label) subset: (n label)) 
		ifTrue:[
			ol := n label - (aNode label).
			ds := aSF indexOf: (aNode label).
			aSF at: (aSF indexOf:(n label)) put: (aNode label).
			aSF at: ds put: (n label).
			n label: (aNode label).
			n next do:[ :ar |
				(ol includes: (ar label))
				ifTrue:[
					ar node prev remove: n ifAbsent:[nil].
					ar node remove: (self nodes).].].
			^true].].].
	^false.!

reuseNode: n with: al with: aSet
	"This method answers an old node if n can be reused and new node
	otherwise."

	| nh nn |
	nh := (n h copy) add: al; yourself.
	self nodes do:[ :on |
		(on h equal: nh)
		ifTrue:[ 
			on prev add: n.
			^on ].].
	nn := HSDagNode newNext: n.
	nn h: nh.
	aSet add: nn.
	self nodes add: nn.
	^nn! !

!HSDag methodsFor: 'public'!

answerDiagnoses
	"Computes diagnosis using self"

	^self topNode answerDiagnoses!

answerDiagnosesProbability
	"Answers the diagnoses computed using fault probabilities."

	^self topNode answerDiagnosesProbability! !

!HSDag methodsFor: 'printing'!

printOn: aStream
	"Prints the hs dag on the given stream"

	self topNode printOn: aStream using: 0! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

HSDag class
	instanceVariableNames: ''!


!HSDag class methodsFor: 'instance creation'!

createHSDag: aTheoremProver with: diagSize with: numberOfDiagnoses
	"Creates a hitting set tree using a diagnosis generator. The diagnosis generator must have implemented
	the method getConflictSet: aComponentSet."

	| result |
	result := self new.
	result createHSDag: aTheoremProver with: diagSize with: numberOfDiagnoses.
	^result!

createHSDagUsingFaultProbabilities: aTheoremProver withDiagSize: aSize withNumberOfDiagnoses: aValue
	"Computes aValue end nodes which are related to diagnoses with highest probability"

	| result |
	result := self new.
	result createHSDagUsingFaultProbabilities: aTheoremProver withDiagSize: aSize withNumberOfDiagnoses: aValue.
	^result!

createHSDagUsingFaultProbabilities: aTheoremProver withNumberOfDiagnoses: aValue
	"Computes aValue end nodes which are related to diagnoses with highest probability"

	| result |
	result := self new.
	result createHSDagUsingFaultProbabilities: aTheoremProver withNumberOfDiagnoses: aValue.
	^result!

new

	^super new initialize! !

!HSDag class methodsFor: 'defaults'!

numberOfDiagnoses
	"Answers the default number of diagnoses"

	^100! !

Object subclass: #HSDagNode
	instanceVariableNames: 'h next prev label probability '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Kernel'!
HSDagNode comment:
'(c) 1995 F. Wotawa TU Wien, Institut fuer Informationssysteme

This class implements the nodes of an hitting set dag. A node has an associated set of
previous nodes, a set of arcs showing to succeding nodes, a label, a history and a
probability. Probability and history can be computed while generating the hs dag. 
Because of performance reasons it is better to store this information inside the node.

'!


!HSDagNode methodsFor: 'label names'!

closeLabel

	^self class closeLabel!

endLabel

	^self class endLabel!

noLabel

	^self class noLabel! !

!HSDagNode methodsFor: 'initialization'!

init

	prev := OrderedCollection new.
	next := OrderedCollection new.
	h := Diagnose new.
	label := nil.
	probability := (1.00).

	^self! !

!HSDagNode methodsFor: 'accessing'!

h

	^h!

h: aSet

	^h := aSet!

label

	^label!

label: aLabel

	^label := aLabel!

next

	^next!

next: aSet

	^next := aSet!

prev

	^prev!

prev: aSet

	^prev := aSet!

probability

	^probability!

probability: aValue

	^probability := aValue! !

!HSDagNode methodsFor: 'printing'!

printOn: aStream using: anInteger

	| txt |

	txt := ''.
	0 to: anInteger do:[:i | txt := txt , '....'.].

	aStream cr; nextPutAll: txt; nextPutAll: 'HSDagNode('.
	self h printOn: aStream.
	aStream nextPutAll: '| '.
	self label printOn: aStream.
	aStream nextPutAll: ') '.
	self next do:[ :arc |
		aStream cr; nextPutAll: txt; nextPutAll: 'HSDagArc('.
		arc label printOn: aStream.
		aStream nextPutAll: ')'.
		arc node printOn: aStream using:(anInteger + 1).].! !

!HSDagNode methodsFor: 'public'!

answerDiagnoses

	| result |

	result := DiagnoseSet new.
	(self label isNil) ifTrue:[ ^result] .
	(self hasEndLabel) ifTrue:[ result add: (self h). ^result ].
	(self hasCloseLabel) ifTrue:[ ^result. ].

	self next do:[ :n | 
		result addAll: (n node answerDiagnoses)].

	^result!

answerDiagnosesProbability

	| result |

	result := DiagnoseSet new.
	self label isNil ifTrue:[ ^result ].
	(self hasEndLabel) ifTrue:[ result add: (self endLabelInfo). ^result ].
	(self hasCloseLabel) ifTrue:[ ^result. ].

	self next do:[ :n | 
		result addAll: (n node answerDiagnosesProbability)].

	^result!

endLabelInfo

	^(self h) convertToDiagnoseProbability: (self probability).!

remove: aSet

	(self prev isEmpty)
	ifTrue:[
		aSet remove: self ifAbsent:[nil].
		self next do:[ :ar |
			ar node prev remove: self ifAbsent:[nil].
			ar node remove: aSet.].].! !

!HSDagNode methodsFor: 'testing'!

hasCloseLabel

	^(label class = Set)
		ifTrue:[ false]
		ifFalse:[ label = self closeLabel ].!

hasEndLabel

	^(label class = Set)
		ifTrue:[ false]
		ifFalse:[ label = self endLabel ].!

hasNoLabel

	^(label class = Set)
		ifTrue:[ false]
		ifFalse:[ label = self noLabel ].! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

HSDagNode class
	instanceVariableNames: ''!


!HSDagNode class methodsFor: 'instance creation'!

new

	^super new init!

newNext: aHSDagNode

	| result |

	result := super new init.
	result prev add: aHSDagNode.

	^result!

newTop

	| result |

	result := super new init.

	^result! !

!HSDagNode class methodsFor: 'label names'!

closeLabel

	^#close!

endLabel

	^#end!

noLabel

	^#no! !

Object subclass: #HSDagArc
	instanceVariableNames: 'node label '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Kernel'!
HSDagArc comment:
'(c) 1995 F. Wotawa TU Wien, Institut fuer Informationssysteme

This class implements the vertexies between two hs dag nodes. It stores
an arc label and the succeding node. '!


!HSDagArc methodsFor: 'accessing'!

label

	^label!

label: anObject

	^label := anObject!

node

	^node!

node: aHSDagNode

	^node := aHSDagNode! !

!HSDagArc methodsFor: 'public'!

node: aHSDagNode label: anObject

	self node: aHSDagNode.
	self label: anObject.

	^self! !

!HSDagArc methodsFor: 'printing'!

printOn: aStream

	aStream nextPutAll: '( '.
	self label printOn: aStream.
	aStream nextPutAll: '->'.
	self	node printOn: aStream.
	aStream nextPutAll: ') '.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

HSDagArc class
	instanceVariableNames: ''!


!HSDagArc class methodsFor: 'instance creation'!

node: aHSDagNode label: anObject

	| result |

	result := super new.
	result node: aHSDagNode label: anObject.

	^result! !

Object subclass: #ComponentModeAssumption
	instanceVariableNames: 'mode component '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Kernel'!


!ComponentModeAssumption methodsFor: 'accessing'!

component

	^component!

component: aComponent

	^component := aComponent!

faultProbability

	^component faultProbabilityValueFor: mode!

mode

	^mode!

mode: aMode

	^mode := aMode!

modes

	^component modes!

nabMode
	"Answer the not abnormal mode of my component."

	^component nabMode!

name

	^component name! !

!ComponentModeAssumption methodsFor: 'testing'!

= aModeAssumption

	^(self mode = (aModeAssumption mode)) and:[ self component = (aModeAssumption component)]!

hash

	^component hash bitXor: (mode hash)! !

!ComponentModeAssumption methodsFor: 'printing'!

printOn: aStream

	mode printOn: aStream.
	aStream nextPutAll: '('.
	component printOn: aStream.
	aStream nextPutAll: ') '.!

shortName

	^(mode asString), '(', (component shortName), ') '! !

!ComponentModeAssumption methodsFor: 'diagnosis methods'!

repair

	^component repairFor: mode!

repairCost

	^component repairCostFor: mode!

resetComponentMode
	"Sets the mode of my component."

	^component resetMode!

setComponentMode
	"Sets the mode of my component."

	^component setMode: mode! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ComponentModeAssumption class
	instanceVariableNames: ''!


!ComponentModeAssumption class methodsFor: 'instance creation'!

mode: aMode component: aComponent

	| cma |
	cma := self new.
	cma mode: aMode.
	cma component: aComponent.
	^cma! !

Diagnose variableSubclass: #ExtendedDiagnosis
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Kernel'!


!ExtendedDiagnosis methodsFor: 'adding'!

addComponent: aDiagComp
	"Adds a diagnosis component to self. The mode is assumed to be the
	components undefined behavior mode."

	^self addComponent: aDiagComp mode: #ab!

addComponent: aDiagComp mode: aMode
	"Adds a diagnosis component  with the given mode to self."
 
	^self add: (ComponentModeAssumption mode: aMode component: aDiagComp).! !

!ExtendedDiagnosis methodsFor: 'set-operations'!

subsumes: aDiagnosis
	"Answers true if self subsumes aDiagnosis.
	We say that self subsumes aDiagnosis if for every component C in self
	the following holds:
	(1) C -> M element of self => C -> M element of aDiagnosis, or
	(2) C -> M element of self => C -> ComponentMode #ab is element of aDiagnosis."

	self do:[ :cma |
		(aDiagnosis includes: cma)
		ifFalse:[ 
			(aDiagnosis includes: (ComponentModeAssumption mode: #ab component: (cma component)))
			ifFalse:[ ^false ].].].
	^true!

withoutOK
	"Answers a new extended diagnosis where all components with a correct
	behavior are eliminated."

	^self select:[ :cma |
		(cma mode = cma nabMode) not ]! !

!ExtendedDiagnosis methodsFor: 'testing'!

= aDiagnosis

	(self size = (aDiagnosis size)) ifFalse:[ ^false ].
	self do:[ :cma |
		(aDiagnosis includes: cma) ifFalse:[ ^false ].].
	^true!

hash

	| hv |
	hv := 17717.
	self do:[ :el | hv := hv bitXor: (el hash) ].
	^hv!

isCloseLabel

	^false!

isEndLabel

	^false!

isNoLabel

	^false!

storesProbabilities

	^false! !

!ExtendedDiagnosis methodsFor: 'diagnosis methods'!

asExtendedChoice

	| choice |
	choice := self species new.
	self do:[ :cma |
		cma modes do:[ :newM |
			(newM = (cma mode))
			ifFalse:[ choice add: (ComponentModeAssumption mode: newM component: (cma component)).].].].
	^choice!

asOrderedCollection
	"Converts self to an ordered collection. Undefined assumptions occure at the end of
	the collection."

	| oc |
	oc := OrderedCollection new.
	self do:[ :cma |
		(cma mode = #ab)
		ifTrue:[ oc addLast: cma ]
		ifFalse:[ oc addFirst: cma ].].
	^oc!

includesComponent: aComp

	self do:[ :cma |
		(cma component = aComp)
		ifTrue:[ ^true ].].
	^false!

onlyComponents
	"Answers a set of components used inside self."

	| result |
	result := self species new.
	self do:[ :cma |
		result add: (cma component).].
	^result!

repair
	"Returns how to repair."

	| result |
	result := OrderedCollection new.
	self do:[ :cma |
		result addLast: (cma repair) ].
	^result!

repairCost
	"Returns the cost for repairing all components given in self."

	| costs |
	costs := 0.0d.
	self do:[ :cma |
		costs := costs + (cma repairCost) ].
	^costs!

resetComponentMode
	"Resets the mode of each component to the default one."

	^self do:[ :cma | cma resetComponentMode ].!

restrictComponents: aCompSet
	"Answers a diagnosis set consisting only out of components which are not element
	of aCompSet."

	| result |
	result := self species new.
	self do:[ :cma |
		(aCompSet includes: cma component)
		ifFalse:[
			result add: cma.].].
	^result!

setComponentMode
	"Sets the mode of each component to the given one."

	^self do:[ :cma | cma setComponentMode ].! !

!ExtendedDiagnosis methodsFor: 'printing'!

printOn: aStream

	(self isEmpty)
	ifTrue:[
		aStream nextPutAll: 'Empty Diagnose'.]
	ifFalse:[
		aStream nextPutAll: '[ '.
		self do:[ :comp |
			aStream nextPutAll: (comp shortName).]
		separatedBy:[ aStream nextPutAll: ', '.].
		aStream nextPutAll: '] '.].! !

!ExtendedDiagnosis methodsFor: 'accessing'!

probability
	"Answers the probability of self. "

	| prob |
	prob := 1.
	self do:[ :cma |
		prob := prob * (cma component faultProbabilityValueFor: (cma mode)).].
	^prob! !

!ExtendedDiagnosis methodsFor: 'converting'!

convertToDiagnose

	^self!

convertToDiagnoseProbability
	"Converts self to an instance of ExtendedDiagnosisProbability."

	| prob ds |
	prob := 1.0d.
	self do:[ :cma |
		ds := cma component diagnosisSystem.
		prob := prob * (cma faultProbability).].
	ds isNil
	ifFalse:[
		((ds diagnosisComponents asSet) - (self onlyComponents asSet)) do:[ :comp |
			prob := prob * (comp faultProbabilityFor: (comp nabMode)).].].
	^self convertToDiagnoseProbability: prob!

convertToDiagnoseProbability: aValue
	"Converts self to an instance of DiagnoseProbability using aValue.
	aValue must be the probability of self."

	| result |
	result := ExtendedDiagnosisProbability new: (self size).
	result addAll: self.
	result probability: aValue.

	^result! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ExtendedDiagnosis class
	instanceVariableNames: ''!


!ExtendedDiagnosis class methodsFor: 'instance creation'!

withOKComponents: aCollection

	| res |
	res := self new.
	aCollection do:[ :comp | res addComponent: comp mode: (comp nabMode).].
	^res! !

!ExtendedDiagnosis class methodsFor: 'examples'!

example1

	"ExtendedDiagnosis example1 inspect"

	| ed1 ed2 c1  c2 c3 txt |

	c1 := DigitalAndGate name: 'A1'.
	c2 := DigitalAndGate name: 'A2'.
	c3 := DigitalAndGate name: 'A3'.

	ed1 := ExtendedDiagnosis 
			with: (ComponentModeAssumption mode: (c1 ok) component: c1)
	"		with: (ComponentModeAssumption mode: (c2 undef) component: c2)"
			with: (ComponentModeAssumption mode: (c3 s0) component: c3).
	ed2 := ExtendedDiagnosis 
			with: (ComponentModeAssumption mode: (c1 ok) component: c1)
			with: (ComponentModeAssumption mode: (c2 undef) component: c2)
			with: (ComponentModeAssumption mode: (c3 undef) component: c3).
	txt := ed1 printString , ' subsumes ', ed2 printString, ' ? ', (ed1 subsumes: ed2) printString.
	^txt! !

ExtendedDiagnosis variableSubclass: #ExtendedDiagnosisProbability
	instanceVariableNames: 'probability '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Diagnosis-Kernel'!


!ExtendedDiagnosisProbability methodsFor: 'printing'!

printOn: aStream

	| ch |

	ch := ''.
	(self isEmpty)
	ifTrue:[
		aStream nextPutAll: 'Empty Diagnose'.]
	ifFalse:[
		self do:[ :comp |
			aStream nextPutAll:ch; nextPutAll: (comp shortName).
			ch := ', '.].
		aStream nextPutAll: (' [ ', (self probability printString), ' ] ').].
	aStream cr.! !

!ExtendedDiagnosisProbability methodsFor: 'accessing'!

probability

	^probability!

probability: aValue

	^probability := aValue! !

!ExtendedDiagnosisProbability methodsFor: 'testing'!

storesProbabilities

	^true! !

!ExtendedDiagnosisProbability methodsFor: 'converting'!

convertToDiagnose
	"Converts self to an instance of ExtendedDiagnosis."

	| result |
	result := ExtendedDiagnosis new: (self size).
	result addAll: self.
	^result!

convertToDiagnoseProbability

	^self!

convertToDiagnoseProbability: aValue

	^self! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ExtendedDiagnosisProbability class
	instanceVariableNames: ''!


!ExtendedDiagnosisProbability class methodsFor: 'instance creation'!

probability: aValue

	| edp |
	edp := self new.
	edp probability: aValue.
	^edp! !

