#!/bin/bash
# (c)2014 Federico Cerutti <federico.cerutti@acm.org> --- MIT LICENCE
# Generic script interface to probo http://sourceforge.net/projects/probo/
# Please feel freet to customize it for your own solver


# CEGARTIX variables
CEGARTIX=${CEGARTIX:='cegartix'}
ORACLE=${ORACLE:='miniSAT'}



# function for echoing on standard error
echoerr()
{
	# to remove standard error echoing, please comment the following line
	echo "$@" 1>&2; 
}

################################################################
# C O N F I G U R A T I O N
# 
# this script must be customized by defining:
# 1) procedure for printing author and version information of the solver
#	(function "information")
# 2) suitable procedures for invoking your solver (function "solver");
# 3) suitable procedures for parsing your solver's output 
#	(function "parse_output");
# 4) list of supported format (array "formats");
# 5) list of supported problems (array "tasks").

# output information
function information
{
	echo "CEGARTIX v10-3-2017 (ICCMA'17 competition version)"
	echo "Wolfgang Dvorak"
	echo "Matti Jarvisalo"
	echo "Johannes P. Wallner (corresponding) <wallner@dbai.tuwien.ac.at>"
# 	echo "Stefan Woltran"
}

# how to invoke your solver: this function must be customized
function solver
{
	fileinput=$1	# input file with correct path

	format=$2	# format of the input file (see below)

	problem=$3    	# problem to solve (see below)

	additional=$4	# additional information, i.e. name of an argument


	# dummy output
	echoerr "input file: $fileinput"
	echoerr "format: $format"
	echoerr "problem: $problem"
	echoerr "additional: $additional"

	if [ "$format" != "apx" ];
	then
		echoerr "unsupported format"
		exit 1
	fi
	
# 	preferred
	if   [ "$problem" = "DS-PR" ];
	then
		$(dirname $0)/$CEGARTIX $fileinput -semantics=pref -mode=skept -argument=$additional -oracle=$ORACLE
	elif [ "$problem" = "EE-PR" ]; 
	then	
		$(dirname $0)/$CEGARTIX $fileinput -semantics=pref -mode=enum -oracle=$ORACLE -depthArg=-1
	elif [ "$problem" = "SE-PR" ]; 
	then	
		$(dirname $0)/$CEGARTIX $fileinput -semantics=pref -mode=enum -extNum=1 -oracle=$ORACLE -depthArg=-1
	elif [ "$problem" = "DC-PR" ];
	then
		$(dirname $0)/$CEGARTIX $fileinput -semantics=pref -mode=cred -argument=$additional -oracle=$ORACLE
# 	semi-stable
	elif   [ "$problem" = "DS-SST" ];
	then
		$(dirname $0)/$CEGARTIX $fileinput -semantics=semi -mode=skept -argument=$additional -oracle=$ORACLE
	elif [ "$problem" = "EE-SST" ]; 
	then	
		$(dirname $0)/$CEGARTIX $fileinput -semantics=semi -mode=enum -oracle=$ORACLE -depthArg=-1
	elif [ "$problem" = "SE-SST" ]; 
	then	
		$(dirname $0)/$CEGARTIX $fileinput -semantics=semi -mode=enum -extNum=1 -oracle=$ORACLE -depthArg=-1
	elif [ "$problem" = "DC-SST" ];
	then
		$(dirname $0)/$CEGARTIX $fileinput -semantics=semi -mode=cred -argument=$additional -oracle=$ORACLE
# 	stage
	elif   [ "$problem" = "DS-STG" ];
	then
		$(dirname $0)/$CEGARTIX $fileinput -semantics=stage -mode=skept -argument=$additional -oracle=$ORACLE
	elif [ "$problem" = "EE-STG" ]; 
	then	
		$(dirname $0)/$CEGARTIX $fileinput -semantics=stage -mode=enum -oracle=$ORACLE -depthArg=-1
	elif [ "$problem" = "SE-STG" ]; 
	then	
		$(dirname $0)/$CEGARTIX $fileinput -semantics=stage -mode=enum -extNum=1 -oracle=$ORACLE -depthArg=-1
	elif [ "$problem" = "DC-STG" ];
	then
		$(dirname $0)/$CEGARTIX $fileinput -semantics=stage -mode=cred -argument=$additional -oracle=$ORACLE
# 	stable
	elif   [ "$problem" = "DS-ST" ];
	then
		$(dirname $0)/$CEGARTIX $fileinput -semantics=stable -mode=skept -argument=$additional -oracle=$ORACLE
	elif [ "$problem" = "EE-ST" ]; 
	then	
		$(dirname $0)/$CEGARTIX $fileinput -semantics=stable -mode=enum -oracle=$ORACLE -depthArg=-1
	elif [ "$problem" = "SE-ST" ]; 
	then	
		$(dirname $0)/$CEGARTIX $fileinput -semantics=stable -mode=enum -extNum=1 -oracle=$ORACLE -depthArg=-1
	elif [ "$problem" = "DC-ST" ];
	then
		$(dirname $0)/$CEGARTIX $fileinput -semantics=stable -mode=cred -argument=$additional -oracle=$ORACLE
# 	complete
	elif   [ "$problem" = "DS-CO" ];
	then
		$(dirname $0)/$CEGARTIX $fileinput -semantics=com -mode=skept -argument=$additional -oracle=$ORACLE
	elif [ "$problem" = "EE-CO" ]; 
	then	
		$(dirname $0)/$CEGARTIX $fileinput -semantics=com -mode=enum -oracle=$ORACLE -depthArg=-1
	elif [ "$problem" = "SE-CO" ]; 
	then	
		$(dirname $0)/$CEGARTIX $fileinput -semantics=grd -mode=enum -extNum=1 -oracle=$ORACLE -depthArg=-1
	elif [ "$problem" = "DC-CO" ];
	then
		$(dirname $0)/$CEGARTIX $fileinput -semantics=com -mode=cred -argument=$additional -oracle=$ORACLE
# 	grounded
	elif   [ "$problem" = "DC-GR" ];
	then
		$(dirname $0)/$CEGARTIX $fileinput -semantics=grd -mode=cred -argument=$additional -oracle=$ORACLE
	elif [ "$problem" = "SE-GR" ]; 
	then	
		$(dirname $0)/$CEGARTIX $fileinput -semantics=grd -mode=enum -extNum=1 -oracle=$ORACLE -depthArg=-1
# 	ideal
	elif   [ "$problem" = "DC-ID" ];
	then
		$(dirname $0)/$CEGARTIX $fileinput -semantics=ideal -mode=cred -argument=$additional -oracle=$ORACLE
	elif [ "$problem" = "SE-ID" ]; 
	then	
		$(dirname $0)/$CEGARTIX $fileinput -semantics=ideal -mode=enum -extNum=1 -oracle=$ORACLE -depthArg=-1
	elif [ "$problem" = "D3" ]; 
	then	
		$(dirname $0)/$CEGARTIX $fileinput -semantics=dung -mode=enum -oracle=$ORACLE -depthArg=-1
	else
		echoerr "unsupported problem"
		exit 1
	fi
}


# how to parse the output of your solver in order to be compliant with probo:
# this function must be customized
# probo accepts solutions of the form:
#	[arg1,arg2,...,argN]       for 
#                                  1. some extension enumeration (SE)
#                                  2. enum. arguments credulously inferred (EC)
#                                  3. enum. arguments skeptically inferred (ES) 
#	[[arg1,arg2,...,argN],[...],...] for extension(s) enumeration
#	YES/NO                           for decision problems
function parse_output
{
	problem=$1
	output="$2"

# 	echoerr "original output: $output"

	#example of parsing for ArgSemSATv0.2, which returns "{arg1,arg2,...}\n{...}\n..."
# 	if [ "$problem" = "EE-PR" ];
# 	then		echo "TEST"
# 		echo -n "["
# 		echo $output | sed 's/{/[/g' | sed 's/}/]/g' | tr -d '\n' \
# 			| sed 's/}{/},{/g' | sed 's/\ /,/g'
# 		echo "]"
# 	else
# 		echoerr "unsupported format or problem"
# 		exit 1
# 	fi
# 	if [ "$problem" = "DS-PR" ] || [ "$problem" = "DS-SST" ] || [ "$problem" = "DS-STG" ] || [ "$problem" = "DC-SST" ] || [ "$problem" = "DS-STG" ];
# 	then
# 		echo $output #| grep -o "ACCEPT\|REJECT" | sed 's/ACCEPT/YES/g' | sed 's/REJECT/NO/g' 
	if [ "$problem" = "SE-ST" ];
	then
		if [ "$output" == "[]" ];
		then
		    echo "NO"
		else
		    echo $output | sed 's/\[\[/\[/g' | sed 's/\]\]/\]/g'
		fi
	elif [ "$problem" = "SE-PR" ] || [ "$problem" = "SE-SST" ] || [ "$problem" = "SE-STG" ] || [ "$problem" = "SE-CO" ] || [ "$problem" = "SE-GR" ] || [ "$problem" = "SE-ID" ];
	then
		echo $output | sed 's/\[\[/\[/g' | sed 's/\]\]/\]/g' #"${output:1: -1}" #
# 		echo $output #| sed 's/[[/[/' | sed 's/]]/]/g' 
# 	elif [ "$problem" = "EE-PR" ] || [ "$problem" = "EE-SST" ] || [ "$problem" = "EE-STG" ];
# 	then
# 		echo -n "["
# 		echo -n $output | sed 's/,\]/\]/g' | sed 's/terminated.*//' | sed 's/\]\[/\],\[/g'
# 		echo "]"
	else 
		echo $output
	fi
}

# accepted formats: please comment those unsupported
formats[1]="apx" # "aspartix" format
# formats[2]="cnf" # conjunctive normal form
# formats[3]="tgf" # trivial graph format

# problems that can be accepted: please comment those unsupported

#+---------------------------------------------------------------------+
#|         I C C M A   '1 5   L I S T   O F   P R O B L E M S          |
#|                                                                     |
tasks[1]="DC-CO" 	# Decide credulously according to Complete semantics
tasks[2]="DS-CO" 	# Decide skeptically according to Complete semantics
tasks[3]="SE-CO" 	# Enumerate some extension according to Complete semantics
tasks[4]="EE-CO" 	# Enumerate all the extensions according to Complete semantics
tasks[5]="DC-PR" 	# Decide credulously according to Preferred semantics
tasks[6]="DS-PR" 	# Decide skeptically according to Preferred semantics
tasks[7]="SE-PR" 	# Enumerate some extension according to Preferred semantics
tasks[8]="EE-PR" 	# Enumerate all the extensions according to Preferred semantics
tasks[9]="DC-ST" 	# Decide credulously according to Stable semantics
tasks[10]="DS-ST" 	# Decide skeptically according to Stable semantics
tasks[11]="SE-ST" 	# Enumerate some extension according to Stable semantics
tasks[12]="EE-ST" 	# Enumerate all the extensions according to Stable semantics
tasks[13]="DC-SST" 	# Decide credulously according to Semi-stable semantics
tasks[14]="DS-SST" 	# Decide skeptically according to Semi-stable semantics
tasks[15]="SE-SST" 	# Enumerate some extension according to Semi-stable semantics
tasks[16]="EE-SST" 	# Enumerate all the extensions according to Semi-stable semantics
tasks[17]="DC-STG" 	# Decide credulously according to Stage semantics
tasks[18]="DS-STG" 	# Decide skeptically according to Stage semantics
tasks[19]="EE-STG" 	# Enumerate all the extensions according to Stage semantics
tasks[20]="SE-STG" 	# Enumerate some extension according to Stage semantics
tasks[21]="DC-GR" 	# Decide credulously according to Grounded semantics
tasks[22]="SE-GR" 	# Enumerate some extension according to Grounded semantics
tasks[23]="DC-ID" 	# Decide credulously according to Ideal semantics
tasks[24]="SE-ID" 	# Enumerate some extension according to Ideal semantics
tasks[25]="D3"          # Dung's Triathlon
#|                                                                     |
#|  E N D   O F   I C C M A   '1 5   L I S T   O F   P R O B L E M S   |
#+---------------------------------------------------------------------+

# tasks[17]="DC-ADM" 	# Decide credulously according to admissiblity
# tasks[18]="DC-CF2" 	# Decide credulously according to CF2 semantics
# tasks[19]="DC-CF" 	# Decide credulously according to conflict-freeness
# tasks[20]="DC-ID" 	# Decide credulously according to Ideal semantics
# tasks[21]="DC-SST" 	# Decide credulously according to Semi-stable semantics
# tasks[22]="DC-STG" 	# Decide credulously according to Stage semantics
# tasks[23]="DS-ADM" 	# Decide skeptically according to admissiblity
# tasks[24]="DS-CF2" 	# Decide skeptically according to CF2 semantics
# tasks[25]="DS-CF" 	# Decide skeptically according to conflict-freeness
# tasks[26]="DS-ID" 	# Decide skeptically according to Ideal semantics
# tasks[27]="DS-SST" 	# Decide skeptically according to Semi-stable semantics
# tasks[28]="DS-STG" 	# Decide skeptically according to Stage semantics
# tasks[29]="EC-ADM" 	# Enumerate all the arguments credulously inferred according to admissiblity
# tasks[30]="EC-CF2" 	# Enumerate all the arguments credulously inferred according to CF2 semantics
# tasks[31]="EC-CF" 	# Enumerate all the arguments credulously inferred according to conflict-freeness
# tasks[32]="EC-CO" 	# Enumerate all the arguments credulously inferred according to Complete semantics
# tasks[33]="EC-GR" 	# Enumerate all the arguments credulously inferred according to Grounded semantics
# tasks[34]="EC-ID" 	# Enumerate all the arguments credulously inferred according to Ideal semantics
# tasks[35]="EC-PR" 	# Enumerate all the arguments credulously inferred according to Preferred semantics
# tasks[36]="EC-SST" 	# Enumerate all the arguments credulously inferred according to Semi-stable semantics
# tasks[37]="EC-STG" 	# Enumerate all the arguments credulously inferred according to Stage semantics
# tasks[38]="EC-ST" 	# Enumerate all the arguments credulously inferred according to Stable semantics
# tasks[39]="EE-ADM" 	# Enumerate all the extensions according to admissiblity
# tasks[40]="EE-CF2" 	# Enumerate all the extensions according to CF2 semantics
# tasks[41]="EE-CF" 	# Enumerate all the extensions according to conflict-freeness
# tasks[42]="EE-ID" 	# Enumerate all the extensions according to Ideal semantics
# tasks[43]="EE-SST" 	# Enumerate all the extensions according to Semi-stable semantics
# tasks[44]="EE-STG" 	# Enumerate all the extensions according to Stage semantics
# tasks[45]="ES-ADM" 	# Enumerate all the arguments skeptically inferred according to admissiblity
# tasks[46]="ES-CF2" 	# Enumerate all the arguments skeptically inferred according to CF2 semantics
# tasks[47]="ES-CF" 	# Enumerate all the arguments skeptically inferred according to conflict-freeness
# tasks[48]="ES-CO" 	# Enumerate all the arguments skeptically inferred according to Complete semantics
# tasks[49]="ES-GR" 	# Enumerate all the arguments skeptically inferred according to Grounded semantics
# tasks[50]="ES-ID" 	# Enumerate all the arguments skeptically inferred according to Ideal semantics
# tasks[51]="ES-PR" 	# Enumerate all the arguments skeptically inferred according to Preferred semantics
# tasks[52]="ES-SST" 	# Enumerate all the arguments skeptically inferred according to Semi-stable semantics
# tasks[53]="ES-STG" 	# Enumerate all the arguments skeptically inferred according to Stage semantics
# tasks[54]="ES-ST" 	# Enumerate all the arguments skeptically inferred according to Stable semantics
# tasks[55]="SE-ADM" 	# Enumerate some extension according to admissiblity
# tasks[56]="SE-CF2" 	# Enumerate some extension according to CF2 semantics
# tasks[57]="SE-CF" 	# Enumerate some extension according to conflict-freeness
# tasks[58]="SE-ID" 	# Enumerate some extension according to Ideal semantics
# tasks[59]="SE-SST" 	# Enumerate some extension according to Semi-stable semantics
# tasks[60]="SE-STG" 	# Enumerate some extension according to Stage semantics

# E N D   O F   C O N F I G U R A T I O N    S E C T I O N
#################################################################

function list_output
{
	declare -a arr=("${!1}")
	check_something_printed=false
	echo -n '['
	for i in ${arr[@]}; 
	do
		if [ "$check_something_printed" = true ];
		then
			echo -n ", "
		fi
		echo -n $i
		check_something_printed=true
	done
        echo ']'
}

function main
{
	if [ "$#" = "0" ]
	then
		information
		exit 0
	fi

	local local_problem=""
	local local_fileinput=""
	local local_format=""
	local local_additional=""

	while [ "$1" != "" ]; do
	    case $1 in
	      "--formats")
		list_output formats[@]
		exit 0
		;;
	      "--problems")
		list_output tasks[@]
		exit 0
		;;
	      "-p")
		shift
		local_problem=$1
		;;
	      "-f")
		shift
		local_fileinput=$1
		;;
	      "-fo")
		shift
		local_format=$1
		;;
	      "-a")
		shift
		local_additional=$1
		;;
	    esac
	    shift
	done

	res=$(solver $local_fileinput $local_format $local_problem $local_additional)

	parse_output $local_problem "$res"
}

main $@
exit 0
