import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;

public class GenerateGraphTreeWidth {
		private static Random random = new Random();

		/**
		 * Generates a graph that is an induced subgraph of a complete grid (4-neighbourhood)
		 * (guarantees an upper bound for the tree-width)
		 *
		 * @param n #nodes in the graph
		 * @param treeWidth maximal tree-width of the graph
		 * @param p probability that a possible (directed) edge is in the graph
		 * @return
		 */
		private static Graph<Integer> generateGraphG4(int n,int treeWidth, double p){
			Graph<Integer> graph=new Graph<Integer>();
			int j;
			for(int i=0;i<n;i++){
				graph.addNode(i);

				j=i-treeWidth;
				if (j>=0){
					if( random.nextDouble()<=p ) graph.addEdge(i, j);
					if( random.nextDouble()<=p ) graph.addEdge(j, i);
				}

				j=i-1;
				if (i % treeWidth != 0){
					if( random.nextDouble()<=p ) graph.addEdge(i, j);
					if( random.nextDouble()<=p ) graph.addEdge(j, i);
				}


			}
			return graph;
		}
		
		/**
		 * Generates a graph that is an induced subgraph of a complete grid (8-neighbourhood)
		 * (guarantees an upper bound for the tree-width)
		 *
		 * @param n #nodes in the graph
		 * @param treeWidth maximal tree-width of the graph
		 * @param p probability that a possible (directed) edge is in the graph
		 * @return
		 */
		private static Graph<Integer> generateGraphG8(int n,int treeWidth, double p){
			Graph<Integer> graph=new Graph<Integer>();
			int k= treeWidth-1;
			
			int j;
			for(int i=0;i<n;i++){
				graph.addNode(i);

				j=i-k; // lower edge 
				if (j>=0){
					if( random.nextDouble()<=p ) graph.addEdge(i, j);
					if( random.nextDouble()<=p ) graph.addEdge(j, i);
				}

				j=i-1; //left edge
				if (i % k != 0){
					if( random.nextDouble()<=p ) graph.addEdge(i, j);
					if( random.nextDouble()<=p ) graph.addEdge(j, i);
		
					j=i-k-1; //left lower edge
					if(j>=0){
						if( random.nextDouble()<=p ) graph.addEdge(i, j);
						if( random.nextDouble()<=p ) graph.addEdge(j, i);
					}
				}			
				j=i-k+1; //right lower edge
				if (j % k != 0 && j>=0){
					if( random.nextDouble()<=p ) graph.addEdge(i, j);
					if( random.nextDouble()<=p ) graph.addEdge(j, i);
				}


			}
			return graph;
		}

		
		/**
		 * Generates a grid with random orientation (8-neighbourhood)
		 * @param n #nodes in the graph
		 * @param treeWidth tree-width of the graph
		 * @param p probability that an arbitrary edge is symmetric
		 * @return
		 */
		private static Graph<Integer> generateGridG8(int n,int treeWidth, double p){
			Graph<Integer> graph=new Graph<Integer>();
			int k= treeWidth-1;
			
			int j;
			for(int i=0;i<n;i++){
				graph.addNode(i);

				j=i-k; // lower edge 
				if (j>=0){
					if( random.nextDouble()<p) {
						graph.addEdge(i, j);
						graph.addEdge(j, i);
					} else{
						if( random.nextDouble()<1/2) {
							graph.addEdge(i, j);
						} else{
							graph.addEdge(j, i);
						}
					}
				}

				j=i-1; //left edge
				if (i % k != 0){
					if( random.nextDouble()<p) {
						graph.addEdge(i, j);
						graph.addEdge(j, i);
					} else{
						if( random.nextDouble()<1/2) {
							graph.addEdge(i, j);
						} else{
							graph.addEdge(j, i);
						}
					}
		
					j=i-k-1; //left lower edge
					if (j>=0){
						if( random.nextDouble()<p) {
							graph.addEdge(i, j);
							graph.addEdge(j, i);
						} else{
							if( random.nextDouble()<1/2) {
								graph.addEdge(i, j);
							} else{
								graph.addEdge(j, i);
							}
						}
					}
				}			
				j=i-k+1; //right lower edge
				if (j % k != 0 && j>=0){
					if( random.nextDouble()<p) {
						graph.addEdge(i, j);
						graph.addEdge(j, i);
					} else{
						if( random.nextDouble()<1/2) {
							graph.addEdge(i, j);
						} else{
							graph.addEdge(j, i);
						}
					}
				}


			}
			return graph;
		}
		
		/**
		 * Generates a grid with random orientation (4-neighbourhood)
		 * @param n #nodes in the graph
		 * @param treeWidth tree-width of the graph
		 * @param p probability that an arbitrary edge is symmetric
		 * @return
		 */
		private static Graph<Integer> generateGridG4(int n,int treeWidth, double p){
			Graph<Integer> graph=new Graph<Integer>();
			int j;
			for(int i=0;i<n;i++){
				graph.addNode(i);

				j=i-treeWidth;
				if (j>=0){
					if( random.nextDouble()<p) {
						graph.addEdge(i, j);
						graph.addEdge(j, i);
					} else{
						if( random.nextDouble()<1/2) {
							graph.addEdge(i, j);
						} else{
							graph.addEdge(j, i);
						}
					}
				}

				j=i-1;
				if (i % treeWidth != 0){
					if( random.nextDouble()<p) {
						graph.addEdge(i, j);
						graph.addEdge(j, i);
					} else{
						if( random.nextDouble()<1/2) {
							graph.addEdge(i, j);
						} else{
							graph.addEdge(j, i);
						}
					}
				}


			}
			return graph;
		}
		/**
		 * Generate arbitrary Random DiGraph
		 * @param n #vertices
		 * @param p Propability that an arc (i,j) is in the graph, 
		 * 			i.e. the estimated edge density)
		 * @return
		 */
		private static Graph<Integer> generateGraph(int n, double p){
			Graph<Integer> graph=new Graph<Integer>();

			for(int i=0;i<n;i++){
				graph.addNode(i);
			}

			for(int i=0;i<n;i++){
				for(int j=0;j<n;j++){
					if(i!=j & random.nextDouble()<p){
						graph.addEdge(i, j);
					}
				}
			}

			return graph;
		}
		
		private static String compargGraph(Graph<Integer> graph){
			StringBuffer stringBuffer= new StringBuffer();

			for(Graph.Node<Integer> node : graph.getNodes()){
				stringBuffer.append(node.getObject().toString());
				for(Graph.Node<Integer> child: node.getChildren()){
					stringBuffer.append(" "+child.getObject().toString());
				}
				stringBuffer.append(",\n");
			}

			return stringBuffer.toString();
		}

		private static String aspartixGraph(Graph<Integer> graph){
			StringBuffer stringBuffer= new StringBuffer();

			for(Graph.Node<Integer> node : graph.getNodes()){
				stringBuffer.append("arg("+node.getObject().toString()+").\n");
				for(Graph.Node<Integer> child: node.getChildren()){
					stringBuffer.append("att("+node.getObject().toString()+","+child.getObject().toString()+").\n");
				}
			}

			return stringBuffer.toString();
		}

		public static String pretty(Integer zahl){
			String hilf="000000"+zahl.toString();
			return hilf.substring(hilf.length()-7);
		}



		public static void main(String[] args) throws Exception {
		
			if (args.length > 3 || args.length < 2) {
				System.out.println("Usage: with 2 arguments: n p ");
				System.out.println("java GenerateGraphTreeWidth n nodes,p prob that arb edge is symmetric (example: 10 0.2");
				System.out.println("Usage: with 3 arguments: n m p");
				System.out.println("java GenerateGraphTreeWidth n nodes, m dimension of grid , p prob edge in grid is mutual attack (otherwise 50 percent chance for each direction). example: 10 3 0.2");
				System.out.println("Note: no self attacks are generated");
			}
// 			if (args.length != 3) {
// 				System.out.println("args not correct");//"Give two args: java GenerateGraphTreeWidth n nodes,p prob that arb edge is symmetric");
// 				System.exit(1);
// 			}

			//System.out.println(aspartixGraph(generateGraph(Integer.parseInt(args[0]),Double.parseDouble(args[1]))));	
			if (args.length == 2) {
				System.out.println(aspartixGraph(generateGraph(Integer.parseInt(args[0]),Double.parseDouble(args[1]))));
			}
			
			if (args.length == 3) {
				System.out.println(aspartixGraph(generateGridG8(Integer.parseInt(args[0]),Integer.parseInt(args[1]),Double.parseDouble(args[2]))));
			}
			//System.out.println(aspartixGraph(generateGridG4(300,7, 0.3)));
		
			//System.out.println(aspartixGraph(generateGraphG8(300,7, 0.3)));			
			
			//System.out.println(aspartixGraph(generateGridG8(1500,7, 0)));	
			
			/*int hibernate_sequence=0;
			String path="/home/dvorak/Desktop/test";
			BufferedWriter out;
			Set<Integer> prob = new HashSet<Integer>();
			prob.add(1);
			prob.add(3);
			prob.add(5);
			prob.add(7);
			out = new BufferedWriter(new FileWriter(path+".txt"));
			for(Integer p:prob){

				for(int n=10;n<=50;n+=10){
					for(int k=0; k<2;k++){
						out.write("#DBAI"+ pretty(hibernate_sequence) +"\n");
						out.write( compargGraph(generateGraph(n,2, p)));
						hibernate_sequence++;
					}
				}
			}
			out.close();*/
		}
}
