/**
 *
 * jQuery.imageChanger
 *
 *     Simple image switcheroo in place, requires specific HTML structure:
 *       .imageChanger-box
 *         .imageChanger-content
 *           .imageChanger-item [one or more]
 *     Navigation HTML is appended to the imageChanger-box.
 *         .imageChanger-navigation
 *     Unfortunately, this solution has a lot of hard-coding, rigidity.
 *     Navigation expects arrow images for previous and next.
 *           .imageChanger-navNext img
 *           .imageChanger-navPrev img
 *
 */

(function($) {

	$.fn.extend({
		imageChanger: function( settings ) {
			var self = this;

			this.settings = $.extend({}, $.fn.imageChanger.defaults, settings);

			this.settings.navigationHtml = [
				'<div class="imageChanger-navigation">',
					'<span class="imageChanger-navPrev"><img width="',
					self.settings.previousButton.width,
					'" height="',
					self.settings.previousButton.height,
					'" border="0" title="Click to view previous image" alt="Previous image" src="',
					self.settings.previousButton.disabled,
					'"><\/span>',
					'<span class="imageChanger-navText"><\/span>',
					'<span class="imageChanger-navNext"><img width="',
					self.settings.nextButton.width,
					'" height="',
					self.settings.nextButton.height,
					'" border="0" title="Click to view next image" alt="Next image" src="',
					self.settings.nextButton.disabled,
					'"><\/span>',
				'<\/div>'
			].join('');

			/**
			 * Makes some imageChanger magic with some DIVs.
			 * Requires a box, with content containing items, and navigation.
			 *
			 * @param {Object} box       A jQuery object for the imageChanger container.
			 * @param {String} direction An optional parameter, next or prev.
			 */
			this.changeImage = function ( box, direction ) {
				var items = box.find('.imageChanger-content .imageChanger-item');
				var selectedItem = items.filter('.selected');
				var selectedNumber = items.index( selectedItem[0] ) + 1;
				var totalNumber = items.size();

				switch ( direction ) {
					case '-1':
					case 'prev':
						if ( selectedNumber > 1 ) {
							selectedItem.fadeOut('fast', function () {
								$(this).removeClass( 'selected' );
								$(this).prev()
									.fadeIn('fast').addClass( 'selected' );
							});
							selectedNumber--;
						}
						break;
					case '1':
					case 'next':
						if ( selectedNumber < totalNumber ) {
							selectedItem.fadeOut('fast', function () {
								$(this).removeClass( 'selected' );
								$(this).next()
									.fadeIn('fast').addClass( 'selected' );
							});
							selectedNumber++;
						}
						break;
				}

				if ( selectedNumber ) {

					// update nav text
					box.find('.imageChanger-navigation .imageChanger-navText')
						.text( selectedNumber + '/' + totalNumber )
					;

					// update previous button
					var previousButton = box.find('.imageChanger-navigation .imageChanger-navPrev');

					// unbind events
					previousButton.unbind();

					if ( selectedNumber == 1 ) {
						// disable previous button
						previousButton
							.addClass('disabled')
							.find( 'img' ).attr( 'src',
								self.settings.previousButton.disabled )
						;
					} else {
						// enable previous button
						previousButton
							.removeClass('disabled')
							.find( 'img' ).attr( 'src',
								self.settings.previousButton.enabled ).end()
							.hover(
								function () {
									$(this).find( 'img' ).attr( 'src',
										self.settings.previousButton.hover );
								},
								function () {
									$(this).find( 'img' ).attr( 'src',
										self.settings.previousButton.enabled );
								}
							)
							.click( function () {
								self.changeImage( box, 'prev' );
								return false;
							})
						;
					}

					// update next button
					var nextButton = box.find('.imageChanger-navigation .imageChanger-navNext');

					// unbind events
					nextButton.unbind();

					if ( selectedNumber == totalNumber ) {
						// disable next button
						nextButton
							.addClass('disabled')
							.find( 'img' ).attr( 'src',
								self.settings.nextButton.disabled )
						;
					} else {
						// enable next buttons
						nextButton
							.removeClass('disabled')
							.find( 'img' ).attr( 'src',
								self.settings.nextButton.enabled ).end()
							.hover(
								function () {
									$(this).find( 'img' ).attr( 'src',
										self.settings.nextButton.hover );
								},
								function () {
									$(this).find( 'img' ).attr( 'src',
										self.settings.nextButton.enabled );
								}
							)
							.click( function () {
								self.changeImage( box, 'next' );
								return false;
							})
						;
					}

				}
			};

			return this.each(function () {
				var items = $(this).find('.imageChanger-content .imageChanger-item');

				// do we have at least one image?
				if ( items.length ) {

					// show box
					$(this).show();

					// show first image
					items.eq(0).show().addClass( 'selected' );

					// do we have more than just one image?
					if ( items.length > 1 ) {

						// create navigation
						$(this).append( self.settings.navigationHtml );

						// show navigation
						$(this).find('.imageChanger-navigation').show();

						// update nav text
						self.changeImage( $(this) );
					}
				}
			});

		}
	});

	// define global defaults, editable by client
	$.fn.imageChanger.defaults = {
		previousButton: {
			width: '27',
			height: '19',
			enabled: 'http://i.cdn.turner.com/money/.element/img/2.0/fortune/buttons/image_prev.gif',
			hover: 'http://i.cdn.turner.com/money/.element/img/2.0/fortune/buttons/image_prev_over.gif',
			disabled: 'http://i.cdn.turner.com/money/.element/img/2.0/fortune/buttons/image_prev_off.gif'
		},
		nextButton: {
			width: '27',
			height: '19',
			enabled: 'http://i.cdn.turner.com/money/.element/img/2.0/fortune/buttons/image_next.gif',
			hover: 'http://i.cdn.turner.com/money/.element/img/2.0/fortune/buttons/image_next_over.gif',
			disabled: 'http://i.cdn.turner.com/money/.element/img/2.0/fortune/buttons/image_next_off.gif'
		}
	};

})(jQuery);
